//==============================================================================
//
// CPVanity
//
// created March 2010, by Luc Pattyn
//
//==============================================================================
// published on CodeProject, see
//	http://www.codeproject.com/KB/dotnet/CPVanity.aspx
//==============================================================================
//
// HISTORY
//	2.4 (09-JUN-2011)
//		- WebClient code removed again, it wasn't dealing with the proper "code page",
//		  resulting in bad characters showing up in some member names.
//		- added CP Vanity label click
//		- added rep graph zoom (click for full screen)
//	2.3 (24-MAY-2011)
//		- bug fix: number of votes/bookmarks/downloads did not understand thousand separator,
//		  so they were wrong when exceeding 999.
//		- add more member information: number of messages, questions, answers, comments;
//		  (visible by hitting SHIFT key, or enlarging the Form).
//		- CPsite now uses WebClient, which seems faster, however:
//			(1) proxy code has moved and changed, and was not tested!
//			(2) the 15 second timeout hasn't been implemented
//	2.2 (27-MAR-2011)
//		- more page sets
//		- added almost a day to make TOTAL the top line for sure
//		- added article downloads
//	2.1 (25-DEC-2010)
//		- fixed width for Participant and Total columns
//		- fixed to cope with another web page change (article views/bookmarks/updated)
//		- added bookmarks tooltip
//		- added category to form title
//	2.0 (06-DEC-2010)
//		- automatic GO when tab first clicked
//		- new Who's Who format
//		- dgvMembers details:
//			= oneMember.Number in green background
//			= ALT/GO excludes Debator from Total rep
//			= added PlatinumCount column
//		- added support for Who's Who pages by reputation
//		- added combobox choosing fetch jobs
//	1.8 (09-OCT-2010)
//		- check for negative RowNumber in all DGV handlers
//		- add a total color representing the fourth highest color (out of 7 rep categories
//		  when GO is pressed with CTRL key down
//		- in highest achievers: sort rep descending initially, and undo scrolling on each sort
//		- dgvMembers: speed-up by PK column, after Sort() dtMembers modified with RaiseEvents false
//		- bookmarks added to dgvArticles
//	1.7 (03-AUG-2010)
//		- adapted to new class names in Who;s Who pages (CP changed 03-AUG-2010)
//	1.6
//		- Highest Achievers now fetching 5+5 who's who pages
//		- Form title contains memberID or "TOP"
//		- added individual member type icons
//	1.5 (06-APR-2010)
//		- removed total rep color
//		- fixed memberID problem (&nbsp; no longer present, now checking for spaces and more)
//		- default article sort ordered reversed
//		- added blink checkbox and registry key
//		- added MVP column
//	1.4 (29-MAR-2010)
//		- CP moved the TOTAL value and dropped its color; rather than chasing after it,
//		  I now calculate them when not in their original location.
//	1.3 (28-MAR-2010)
//		- fixed an occasional bug about dbgMembers.Columns["Name"] not existing while
//		  reloading the top achievers.
//		- fixed bronze color
//		- stopped setting a white background on empty cells; this fixed the DGV repaint
//			problems, and allowed me to remove the earlier workaround attempts (which
//			failed miserably when scrolling)gchan.
//	1.2 (23-MAR-2010)
//		- fixed a bug with duplicate article titles (thanks Sacha!)
//		- added try-catch to survive and log the sporadic sort-on-column1 problems
//	1.1 (23-MAR-2010)
//		- fixed culture bug (use InvariantCulture for CP numbers)
//		- added tooltip to DGVs
//		- set both DGV to read-only
//		- changed column 1 in both DGV into DataGridViewLinkColumn
//		- replaced article URL by article title
//		- added DGV repaint on Form activation
//		- some minor cosmetic changes
//	1.0 (22-MAR-2010)
//		original version
//
//==============================================================================
//
// TODO
//
// UNRESOLVED
//	- Sandeep Mewara reported 403/forbidden errors hitting downloadPage() line
//		HttpWebResponse resp=(HttpWebResponse)req.GetResponse();
//		status 23-MAR-2010: Can't reproduce, Sandeep should investigate and/or ask Chris
//
// CLASSIFIED
//	- Tom Deketelaere suggests search-by-name
//		status 23-MAR-2010: Would need a better CP interface, I'm not going to
//							use the Who's Who search box!
//	- Sandeep Mewara suggested to exclude the TOTAL row from sorting
//		status 23-MAR-2010: that would require custom sorting. Not planned. Having the
//							total row moving to first position has its advantages too.
//	- a few people reported an ArgumentOutOfRangeException in dgvMembers_CellContentClick
//		when clicking on the Name header;
//		status 23-MAR-2010: not found. However column changed now to LinkColumn. Wait and see.
//
//==============================================================================

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;					// Process
using System.Drawing;
using System.Drawing.Imaging;
using System.Drawing.Text;					// TextRenderingHint
using System.IO;
using System.Runtime.InteropServices;		// DllImport
using System.Text;
using System.Threading;
using System.Windows.Forms;
using System.Xml;
using System.Xml.Serialization;
using Microsoft.Win32;

namespace CPVanity {
	public partial class frmMain : Form {
		private const string VERSION="V2.4";
		private const string COMPANY_LOGO_URL="http://www.perceler.com/images/CPVanity_PTSC21.gif";
		private const string COMPANY_SITE_URL="http://www.perceler.com/";
		private const string CPVANITY_ARTICLE="http://www.codeproject.com/KB/dotnet/CPVanity.aspx";

		private const int DGVA_COL_NUMBER=0;
		private const int DGVA_COL_TITLE=1;
		private const int DGVA_COL_UPDATED=2;
		private const int DGVA_COL_VIEWS=3;
		private const int DGVA_COL_DOWNLOADS=4;
		private const int DGVA_COL_BOOKMARKS=5;
		private const int DGVA_COL_VOTES=6;
		private const int DGVA_COL_RATING=7;
		private const int DGVA_COL_POPULARITY=8;

		private const int DGVM_COL_KEY=0;
		private const int DGVM_COL_NUMBER=1;
		private const int DGVM_COL_NAME=2;
		private const int DGVM_COL_IMAGE=3;
		private const int DGVM_COL_PLATINUMCOUNT=4;
		private const int DGVM_COL_FIRSTREP=5;
		private const int DGVM_COL_REP_AUTHOR=DGVM_COL_FIRSTREP;
		private const int DGVM_COL_REP_AUTHORITY=DGVM_COL_REP_AUTHOR+1;
		private const int DGVM_COL_REP_DEBATOR=DGVM_COL_REP_AUTHORITY+1;
		private const int DGVM_COL_REP_EDITOR=DGVM_COL_REP_DEBATOR+1;
		private const int DGVM_COL_REP_ENQUIRER=DGVM_COL_REP_EDITOR+1;
		private const int DGVM_COL_REP_ORGANISER=DGVM_COL_REP_ENQUIRER+1;
		private const int DGVM_COL_REP_PARTICIPANT=DGVM_COL_REP_ORGANISER+1;
		private const int DGVM_COL_REP_TOTAL=DGVM_COL_REP_PARTICIPANT+1;
		private const int DGVM_COL_LASTREP=DGVM_COL_REP_TOTAL;
		private const int DGVM_COL_FIRSTCOUNTER=DGVM_COL_LASTREP+1;
		private const int DGVM_COL_ARTICLES=DGVM_COL_FIRSTCOUNTER;
		private const int DGVM_COL_TIPS=DGVM_COL_ARTICLES+1;
		private const int DGVM_COL_BLOGS=DGVM_COL_TIPS+1;
		private const int DGVM_COL_MESSAGES=DGVM_COL_BLOGS+1;
		private const int DGVM_COL_QUESTIONS=DGVM_COL_MESSAGES+1;
		private const int DGVM_COL_ANSWERS=DGVM_COL_QUESTIONS+1;
		private const int DGVM_COL_COMMENTS=DGVM_COL_ANSWERS+1;
		private const int DGVM_COL_LASTCOUNTER=DGVM_COL_COMMENTS;

		private uint memberID;
		private Member clickedMember;
		private CPSite CP;
		private Bitmap bmReputation;
		private Bitmap bmCodeProjectLogo;
		private Bitmap bmCompanyLogo;
		private string page;
		private string name;
		private string adornedName;
		private string adornedNameWithHonors;
		private List<string> honors;
		private List<Article> articles;
		private Dictionary<string, string> titlesToUrls;
		private Dictionary<string, Member> members;
		private DataTable dtArticles;
		private DataTable dtMembers;
		private bool storeMemberIDInRegistry;	// gets true when button is clicked
		private BackgroundWorker bgw;
		private bool blinkNowShows;
		private bool withBlink;
		private System.Windows.Forms.Timer blinkTimer=new System.Windows.Forms.Timer();
		private Member champion;
		private Font numberFont=new Font("Courier New", 9, FontStyle.Regular);
		private Font menuFontUnselected;
		private Font menuFontSelected;
		private Color menuDarkGreen=Color.FromArgb(74, 141, 2);
		private Color menuLightGreen=Color.FromArgb(197, 217, 128);
		private Font nameFont;
		private Font linkFont;
		private double columnScale;
		private bool throwUp;
		private long totalViews;
		private int totalVotes;
		private int totalBookmarks;
		private int totalDownloads;
		private bool excludeDebatorFromTotal; // when true, exclude Debator from Total rep
		private int dgvMembersSortColumn=-1;
		private ListSortDirection dgvMembersSortOrder=ListSortDirection.Descending;
		private BindingSource dgvMembersBindingSource;
		private Font articleHeaderFont=new Font("Arial", 9);
		private Font titleFont=new Font("Arial", 9.5f);
		private FetchJob fetchJob;
		private DateTime tobMembersStarted;
		private string membersTitle="Top";

		public frmMain() {
			InitializeComponent();
			// use full screen height
			Height=Screen.PrimaryScreen.WorkingArea.Height-10;
			log("Height="+Height);
			panBanner.BackColor=Color.FromArgb(255, 153, 0);
			lblProgram.Text="CP Vanity version "+VERSION;
			// hide the native TabControl tabs
			panGreen.BackColor=menuDarkGreen;
			tabControl2.Top-=30;
			tabControl2.Height+=30;
			panGreen.BringToFront();
			// save some values we'll need later
			menuFontSelected=lblOneMember.Font;
			menuFontUnselected=lblTopMembers.Font;
			nameFont=dgvMembers.DefaultCellStyle.Font;
			linkFont=new Font("Arial", 9, FontStyle.Underline);
			columnScale=(ClientSize.Width-50)/970.0;
			throwUp=readFromRegistry("ThrowUp")!=0;
			withBlink=readFromRegistry("Blinking")!=0;
			cbBlink.Checked=withBlink;
			log("throwUp="+throwUp);
			cbFetchJob.Items.Add(new FetchJob(1, 0, 0, "1 Reputation Page"));
			cbFetchJob.Items.Add(new FetchJob(2, 0, 0, "2 Reputation Pages"));
			cbFetchJob.Items.Add(new FetchJob(5, 0, 0, "5 Reputation Pages"));
			cbFetchJob.Items.Add(new FetchJob(0, 1, 0, "1 Article Page"));
			cbFetchJob.Items.Add(new FetchJob(0, 2, 0, "2 Article Pages"));
			cbFetchJob.Items.Add(new FetchJob(0, 5, 0, "5 Article Pages"));
			cbFetchJob.Items.Add(new FetchJob(0, 0, 1, "1 Message Page"));
			cbFetchJob.Items.Add(new FetchJob(0, 0, 2, "2 Message Pages"));
			cbFetchJob.Items.Add(new FetchJob(0, 0, 5, "5 Message Pages"));
			cbFetchJob.Items.Add(new FetchJob(0, 5, 5, "5 Art + 5 Msg Pages"));
			cbFetchJob.Items.Add(new FetchJob(5, 5, 5, "5 R + 5 A + 5 M Pages"));
			int fetchJobIndex=(int)readFromRegistry("FetchJob");
			if (fetchJobIndex>=cbFetchJob.Items.Count) fetchJobIndex=0;
			cbFetchJob.SelectedIndex=fetchJobIndex;
			toolTip1.SetToolTip(btnTopMembers, "ALT/Go = exclude Debator from Total rep");
		}

		private class FetchJob {
			public int ReputationPages;
			public int ArticlePages;
			public int MessagePages;
			public int TotalPages;
			public string Text;
			public FetchJob(int r, int a, int m, string text) {
				ReputationPages=r;
				ArticlePages=a;
				MessagePages=m;
				TotalPages=r+a+m;
				Text=text;
			}
			public override string ToString() {
				return Text;
			}
		}

		private void log(string s) {
			s=DateTime.Now.ToString("HH:mm:ss ")+s;
			Console.WriteLine(s);
		}

		private void logSilent(Exception exc) {
			foreach (string s in exc.ToString().Split('\n', '\r')) if (s.Length!=0) log(s);
			if (throwUp) MessageBox.Show(exc.Message, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
		}

		private void logWithMessageBox(Exception exc) {
			foreach (string s in exc.ToString().Split('\n', '\r')) if (s.Length!=0) log(s);
			MessageBox.Show(exc.Message, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
		}

		private void beep() {
			MessageBeep(0x30);
		}
		[DllImport("user32.dll")]
		public static extern bool MessageBeep(int type);

		private void frmMain_Load(object sender, EventArgs e) {
			// accept a member ID from the command line; if none go look in the registry
			string[] args=Environment.GetCommandLineArgs();
			if (args.Length>1) {
				uint.TryParse(args[1], out memberID);
			} else {
				memberID=readFromRegistry("MemberID");
			}
			if (memberID!=0) tbMember.Text=memberID.ToString();
			// run automatically unless CTRL key is pressed
			clearResultsOneMember();
			clearResultsTopMembers();
			if ((Control.ModifierKeys&Keys.Control)==0) getOneMember();
			blinkTimer.Interval=1000;
			blinkTimer.Tick+=(s1, e1) => {
				blinkNowShows=!blinkNowShows & withBlink; 
				if (dgvMembers!=null) dgvMembers.Invalidate();
			};
			blinkTimer.Enabled=withBlink;
			try {
				Bitmap img=new Bitmap(typeof(frmMain), "Resources.MVP.gif");
				TableFactory.MvpImage=new Bitmap(img, 16, 16);
				log("MVP image size="+TableFactory.MvpImage.Size);
			} catch (Exception exc) { logSilent(exc); }
		}

		private void clearResultsOneMember() {
			lblTime.Text="";
			lblArticleCount.Text="";
			lblArticleRating.Text="";
			lblVotingRatio.Text="";
			lblProfilePage.Visible=false;
			lblProfilePage.LinkVisited=false;
			lblProgress.Text="";
			prgBar.Visible=false;
			pbReputation.Image=null;
			toolTip1.SetToolTip(pbReputation, "");
			dgvArticles.Visible=false;
			honors=null;
		}

		private void clearResultsTopMembers() {
			lblTime2.Text="";
			lblMemberCount.Text="";
			lblProgress.Text="";
			prgBar.Visible=false;
			dgvMembers.Visible=false;
			btnShift.Visible=false;
			cbBlink.Visible=false;
			dgvMembersSortColumn=-1;
		}

		private uint readFromRegistry(string name) {
			try {
				RegistryKey key=Registry.CurrentUser;
				key=key.OpenSubKey("Software");
				key=key.OpenSubKey("CPVanity");
				return uint.Parse(key.GetValue(name).ToString());
			} catch {// ignore errors, most likely key does not yet exist
				return 0;
			}
		}

		private void writeToRegistry(string name, uint memberId) {
			try {
				RegistryKey key=Registry.CurrentUser;
				key=key.OpenSubKey("Software", true);
				key=key.CreateSubKey("CPVanity");
				key.SetValue(name, memberId);
			} catch(Exception exc) {
				logSilent(exc);
			}
		}

		// start processing on ENTER
		private void tbMember_KeyDown(object sender, KeyEventArgs e) {
			Keys k=e.KeyCode;
			log("KeyDown: "+k);
			if (k==Keys.Enter) {
				e.Handled=true;
				e.SuppressKeyPress=true;
				getOneMember();
			}
		}

		// accept digits, backspace, CTRL/X, CTRL/V, CTRL/C
		private void tbMember_KeyPress(object sender, KeyPressEventArgs e) {
			char c=e.KeyChar;
			log("KeyPress: "+c+" = 0x"+((int)c).ToString("X2"));
			e.Handled="0123456789\b\x0018\x0016\x0003".IndexOf(e.KeyChar)<0;
			if (e.Handled) beep();
			else storeMemberIDInRegistry=true;
		}

		private void btnOneMember_Click(object sender, EventArgs e) {
			getOneMember();
		}

		private void getOneMember() {
			clearResultsOneMember();
			uint ID=0;
			bool OK=uint.TryParse(tbMember.Text, out ID);
			if (OK) {
				memberID=ID;
				if (storeMemberIDInRegistry) writeToRegistry("MemberID", memberID);
			}
			if (ID==0) {
				Text="CP Vanity";
				wbName.DocumentText=HtmlConverter.CreateDocument(10,
					"Enter a CodeProject member number and click the button");
			} else {
				Text="CP Vanity -- "+memberID.ToString();
				btnOneMember.Enabled=false;
				Cursor=Cursors.WaitCursor;
				CP=new CPSite(memberID);
				lblProgress.Text="Accessing "+CP.BaseURL;
				wbName.DocumentText=HtmlConverter.CreateDocument(10, "");
				lblTime.Text=DateTime.Now.ToString("dd-MMM-yyyy HH:mm");
				prgBar.Value=0;
				prgBar.Visible=true;
				bgw=new BackgroundWorker();
				bgw.WorkerReportsProgress=true;
				bgw.DoWork += new DoWorkEventHandler(bgwOneMember_DoWork);
				bgw.ProgressChanged+=new ProgressChangedEventHandler(bgwOneMember_ProgressChanged);
				bgw.RunWorkerCompleted+=new RunWorkerCompletedEventHandler(bgwOneMember_RunWorkerCompleted);
				bgw.RunWorkerAsync(memberID);
			}
		}

		private void bgwOneMember_DoWork(object sender, DoWorkEventArgs e) {
			bgw.ReportProgress(10);
			CP.Ping();
			bgw.ReportProgress(20);
			bmCompanyLogo=CP.DownloadImage(COMPANY_LOGO_URL);
			bgw.ReportProgress(30, pbCompany);
			bmCodeProjectLogo=CP.GetBob();
			bgw.ReportProgress(40, pbCodeProject);
			bmReputation=CP.GetReputationGraph();
			bgw.ReportProgress(55, pbReputation);
			honors=CP.GetHonors();
			bgw.ReportProgress(65);
			page=CP.GetArticlePage();
			name=CP.GetName();
			adornedName=CP.GetAdornedName();
			if (adornedName.Length==0) adornedName=name;
			adornedNameWithHonors=adornedName;
			if (honors!=null) {
				adornedNameWithHonors+="<br /><br />\n<table><tr>";
				int count=0;
				foreach (string honor in honors) {
					count++;
					if (count%4==0) adornedNameWithHonors+="</tr><tr>";
					adornedNameWithHonors+="<td><img src=\""+honor+"\" /></td>\n";
				}
				adornedNameWithHonors+="</tr></table>\n";
			}
			log("adornedNameWithHonors="+adornedNameWithHonors);
			bgw.ReportProgress(80, wbName);
			articles=CP.GetArticles();
			dtArticles=TableFactory.GetArticlesTable(articles, out totalViews, out totalVotes,
				out totalBookmarks, out totalDownloads);
		}

		void bgwOneMember_ProgressChanged(object sender, ProgressChangedEventArgs e) {
			log("progress="+e.ProgressPercentage+"%");
			if (e.UserState==pbCompany) pbCompany.Image=bmCompanyLogo;
			if (e.UserState==wbName) wbName.DocumentText=HtmlConverter.CreateDocument(10,
				"Failed to access "+CP.BaseURL);
			if (e.UserState==pbCodeProject) pbCodeProject.Image=bmCodeProjectLogo;
			if (e.UserState==pbReputation) {
				pbReputation.Image=bmReputation;
				toolTip1.SetToolTip(pbReputation, "Click to enlarge");
			}
			if (e.UserState==wbName) {
				log("Name="+name);
				log("AdornedName="+adornedName);
				wbName.DocumentText=HtmlConverter.CreateDocument(12, adornedNameWithHonors);
			}
			prgBar.Value=e.ProgressPercentage;
		}

		void bgwOneMember_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e) {
			log("bgwOneMember_RunWorkerCompleted");
			if (e.Error!=null) {
				log("ERROR: "+e.Error.ToString());
				adornedNameWithHonors=e.Error.Message;
			} else {
				lblProfilePage.Visible=true;
				titlesToUrls=new Dictionary<string, string>();
				foreach (Article article in articles) {
					string title=article.Title;
					string upd=article.Updated.ToString("yyyyMMMdd.HHmmss");
					if (title.Length!=0) titlesToUrls[title+upd]=article.URL;
				}
				//are there any articles for the current member
				int artCount=0;
				if (articles!=null) artCount=articles.Count;
				if (artCount!=0) {
					//only worry about getting the rest if the author has articles
					log("Name="+name);
					lblArticleCount.Text=plural(artCount, "article#s available");
					if (artCount>1) lblArticleRating.Text="Average rating: "+CP.GetAverageRating()+" / 5";
					dgvArticles.Columns.Clear();
					dgvArticles.DataSource=dtArticles;
					dgvArticles.Columns.RemoveAt(DGVA_COL_TITLE);
					dgvArticles.Columns.Insert(DGVA_COL_TITLE, new DataGridViewLinkColumn());
					dgvArticles.Columns[DGVA_COL_TITLE].DefaultCellStyle.Font=titleFont;
					dgvArticles.Columns[DGVA_COL_TITLE].DataPropertyName="Title";
					dgvArticles.Columns[DGVA_COL_TITLE].SortMode=DataGridViewColumnSortMode.Automatic;
					dgvArticles.Columns[DGVA_COL_TITLE].Name="Title";
					adjustArticleColumns();
					dgvArticles.Sort(dgvArticles.Columns[DGVA_COL_UPDATED], ListSortDirection.Descending);
					dgvArticles.Visible=true;
					if (totalVotes!=0) {
						long ratio=totalViews/totalVotes;
						lblVotingRatio.Text="Average views per vote: "+ratio;
					}
				} else {//there are no articles to show, so update GUI to show this
					lblArticleCount.Text="No articles found";
					dgvArticles.Visible=false;
				}
			}
			lblProgress.Text="";
			prgBar.Visible=false;
			btnOneMember.Enabled=true;
			Cursor=Cursors.Default;
			wbName.DocumentText=HtmlConverter.CreateDocument(12, adornedNameWithHonors);
		}

		private string plural(int count, string text) {
			if (count==0) text="No "+text.Replace("#s", "s");
			else if (count==1) text="One "+text.Replace("#s", "");
			else text=count.ToString()+" "+text.Replace("#s", "s");
			return text;
		}

		private void adjustArticleColumns() {
			dgvArticles.ColumnHeadersDefaultCellStyle.Font=articleHeaderFont;
			foreach (DataGridViewColumn col in dgvArticles.Columns) {
				col.ReadOnly=true;
			}
			dgvArticles.Columns[DGVA_COL_NUMBER].SortMode=DataGridViewColumnSortMode.NotSortable;
			// set some formatting styles
			dgvArticles.Columns[DGVA_COL_NUMBER].DefaultCellStyle.Format="#,###";		// Number column
			dgvArticles.Columns[DGVA_COL_UPDATED].DefaultCellStyle.Format="dd MMM yyyy";// Updated column
			dgvArticles.Columns[DGVA_COL_VIEWS].DefaultCellStyle.Format="#,###";		// Views column
			dgvArticles.Columns[DGVA_COL_VOTES].DefaultCellStyle.Format="#,###";		// Votes column
			dgvArticles.Columns[DGVA_COL_BOOKMARKS].DefaultCellStyle.Format="#,###";	// Bookmarks column
			dgvArticles.Columns[DGVA_COL_DOWNLOADS].DefaultCellStyle.Format="#,###";	// Downloads column
			// force trailing zeroes, and don't show zero
			dgvArticles.Columns[DGVA_COL_POPULARITY].DefaultCellStyle.Format="0.00;;.";	// Popularity column
			//resize all columns
			dgvArticles.Columns[DGVA_COL_NUMBER].Width=colWid(50); // # column
			dgvArticles.Columns[DGVA_COL_TITLE].Width=colWid(370); // URL column
			dgvArticles.Columns[DGVA_COL_UPDATED].Width=colWid(100); // Updated column
			dgvArticles.Columns[DGVA_COL_VIEWS].Width=colWid(100); // Views column
			dgvArticles.Columns[DGVA_COL_RATING].Width=colWid(70); // Rating column
			dgvArticles.Columns[DGVA_COL_VOTES].Width=colWid(70); // Votes column
			dgvArticles.Columns[DGVA_COL_DOWNLOADS].HeaderText="DL"; // Downloads column
			dgvArticles.Columns[DGVA_COL_DOWNLOADS].Width=colWid(80); // Downloads column
			dgvArticles.Columns[DGVA_COL_BOOKMARKS].HeaderText="BM"; // Bookmarks column
			dgvArticles.Columns[DGVA_COL_BOOKMARKS].Width=colWid(60); // Bookmarks column
			dgvArticles.Columns[DGVA_COL_POPULARITY].Width=colWid(90); // Popularity column
		}

		// adjust column width for different DPI setting
		private int colWid(int wid96) {
			//return (int)(wid96*dpi/96);
			return (int)(wid96*columnScale);
		}

		private void dgvArticles_CellContentClick(object sender, DataGridViewCellEventArgs e) {
			try {
				if (e.ColumnIndex==1) {
					log("dgvArticles_CellContentClick: row="+e.RowIndex+"  col="+e.ColumnIndex);
					object cell=dgvArticles["Title", e.RowIndex].Value;
					string title=cell==null?"":cell.ToString();
					if (title.Length!=0) {
						DateTime updated=(DateTime)dgvArticles["Updated", e.RowIndex].Value;
						string upd=updated.ToString("yyyyMMMdd.HHmmss");
						string url=titlesToUrls[title+upd];
						launchWebPage(url);
					}
				}
			} catch (Exception exc) {
				// there is an occasional exception here, an out-of-bound index
				// still not sure how it occurs
				logSilent(exc);
			}
		}

		// undo any selection
		private void dgv_SelectionChanged(object sender, EventArgs e) {
			dgvArticles.ClearSelection();
			dgvMembers.ClearSelection();
		}

		private void btnTopMembers_Click(object sender, EventArgs e) {
			getTopMembers();
		}

		private void getTopMembers() {
			setMembersTitle("Top");
			clearResultsTopMembers();
			btnTopMembers.Enabled=false;
			Cursor=Cursors.WaitCursor;
			fetchJob=cbFetchJob.SelectedItem as FetchJob;
			writeToRegistry("FetchJob", (uint)cbFetchJob.SelectedIndex);
			CP=new CPSite(0);
			lblProgress.Text="Accessing "+CP.BaseURL;
			//wbName.DocumentText=HtmlConverter.CreateDocument(10, "");
			tobMembersStarted=DateTime.Now;
			lblTime2.Text=tobMembersStarted.ToString("dd-MMM-yyyy HH:mm");
			prgBar.Value=0;
			prgBar.Visible=true;
			bgw=new BackgroundWorker();
			bgw.WorkerReportsProgress=true;
			bgw.DoWork += new DoWorkEventHandler(bgwTopMembers_DoWork);
			bgw.ProgressChanged+=new ProgressChangedEventHandler(bgwTopMembers_ProgressChanged);
			bgw.RunWorkerCompleted+=new RunWorkerCompletedEventHandler(bgwTopMembers_RunWorkerCompleted);
			excludeDebatorFromTotal=(Control.ModifierKeys&Keys.Alt)!=0;
			bgw.RunWorkerAsync(0);
		}

		private void bgwTopMembers_DoWork(object sender, DoWorkEventArgs e) {
			bgw.ReportProgress(5);
			CP.Ping();
			bgw.ReportProgress(9);
			CP.ClearWhoIsWhoPageCounters();
			log("Begin download Who's Who");
			List<string> pages=new List<string>();
			// percentage range better be a multiple of 30 as totalPages is 1/2/5/10/15
			int progress=9;
			int progressDelta=90/fetchJob.TotalPages;
			for (int i=1; i<=fetchJob.ReputationPages; i++) {
				pages.Add(CP.GetWhoIsWhoPage('R', i));
				bgw.ReportProgress(progress+=progressDelta);
			}
			for (int i=1; i<=fetchJob.ArticlePages; i++) {
				pages.Add(CP.GetWhoIsWhoPage('A', i));
				bgw.ReportProgress(progress+=progressDelta);
			}
			for (int i=1; i<=fetchJob.MessagePages; i++) {
				pages.Add(CP.GetWhoIsWhoPage('M', i));
				bgw.ReportProgress(progress+=progressDelta);
			}
			string page=string.Concat(pages.ToArray());
			log("Total size of Who's Who pages: "+(page.Length/1024)+" Kchar");
			bgw.ReportProgress(99);
			members=CP.GetTopMembers(page, excludeDebatorFromTotal);
			// to be sortable the DGV needs an IBindingList
			dtMembers=TableFactory.GetMembersTable(members);
			// determine highest number in each category
			champion=new Member();
			foreach (Member member in members.Values) {
				if (member.Author>champion.Author) champion.Author=member.Author;
				if (member.Authority>champion.Authority) champion.Authority=member.Authority;
				if (member.Debator>champion.Debator) champion.Debator=member.Debator;
				if (member.Editor>champion.Editor) champion.Editor=member.Editor;
				if (member.Enquirer>champion.Enquirer) champion.Enquirer=member.Enquirer;
				if (member.Organiser>champion.Organiser) champion.Organiser=member.Organiser;
				if (member.Participant>champion.Participant) champion.Participant=member.Participant;
				if (member.Total>champion.Total) champion.Total=member.Total;
				if (member.ArticleCount>champion.ArticleCount) champion.ArticleCount=member.ArticleCount;
				if (member.TipCount>champion.TipCount) champion.TipCount=member.TipCount;
				if (member.BlogCount>champion.BlogCount) champion.BlogCount=member.BlogCount;
				if (member.MessageCount>champion.MessageCount) champion.MessageCount=member.MessageCount;
				if (member.QuestionCount>champion.QuestionCount) champion.QuestionCount=member.QuestionCount;
				if (member.AnswerCount>champion.AnswerCount) champion.AnswerCount=member.AnswerCount;
				if (member.CommentCount>champion.CommentCount) champion.CommentCount=member.CommentCount;
			}
		}

		private void bgwTopMembers_ProgressChanged(object sender, ProgressChangedEventArgs e) {
			log("progress="+e.ProgressPercentage+"%");
			prgBar.Value=e.ProgressPercentage;
			if (DateTime.Now>tobMembersStarted.AddSeconds(8)) {
				lblMemberCount.Text="This wouldn't take so long if a CP Web Service were available.";
			}
		}

		private void adjustMemberColumns() {
			dgvMembers.Columns[DGVM_COL_KEY].Visible=false;
			dgvMembers.Columns[DGVM_COL_NUMBER].SortMode=DataGridViewColumnSortMode.NotSortable;
			// set formatting styles
			for (int i=0; i<=DGVM_COL_LASTCOUNTER; i++) dgvMembers.Columns[i].DefaultCellStyle.Format="#,###;#,###;"; // no zero
			//resize all columns
			dgvMembers.Columns[DGVM_COL_NUMBER].Width=colWid(35);			// # column
			dgvMembers.Columns[DGVM_COL_NAME].Width=colWid(230);			// Name column
			dgvMembers.Columns[DGVM_COL_IMAGE].Width=colWid(20);			// image column
			dgvMembers.Columns[DGVM_COL_PLATINUMCOUNT].Width=colWid(20);	// platinum count column
			dgvMembers.Columns[DGVM_COL_PLATINUMCOUNT].SortMode=DataGridViewColumnSortMode.Programmatic;
			dgvMembers.Columns[DGVM_COL_PLATINUMCOUNT].ToolTipText="";
			dgvMembers.Columns[DGVM_COL_PLATINUMCOUNT].DefaultCellStyle.Padding=new Padding(0);
			dgvMembers.Columns[DGVM_COL_KEY].Frozen=true;
			dgvMembers.Columns[DGVM_COL_NUMBER].Frozen=true;
			dgvMembers.Columns[DGVM_COL_NAME].Frozen=true;
			for (int i=DGVM_COL_FIRSTREP; i<=DGVM_COL_LASTREP; i++) {
				int width=83;
				if (i==DGVM_COL_REP_PARTICIPANT) width+=2;
				if (i==DGVM_COL_REP_TOTAL) width-=2;
				if (dgvMembers.Columns[i].HeaderText.Length>7) width+=7;
				dgvMembers.Columns[i].Width=colWid(width); // rep columns
				dgvMembers.Columns[i].SortMode=DataGridViewColumnSortMode.Programmatic;
			}
			for (int i=DGVM_COL_FIRSTCOUNTER; i<=DGVM_COL_LASTCOUNTER; i++) {
				string name=dgvMembers.Columns[i].Name;
				int width=100;
				if (name.Length<=5) width=75;
				dgvMembers.Columns[i].Width=colWid(width);
				dgvMembers.Columns[i].SortMode=DataGridViewColumnSortMode.Programmatic;
			}
			foreach (DataGridViewColumn col in dgvMembers.Columns) col.ReadOnly=true;
		}

		private void bgwTopMembers_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e) {
			log("bgwTopMembers_RunWorkerCompleted");
			lblProgress.Text="";
			if (e.Error!=null) {
				log("ERROR: "+e.Error.ToString());
			} else {
				//are there any articles for the current member
				int memberCount=0;
				if (members!=null) memberCount=members.Count;
				if (memberCount!=0) {
					int mvpCount=0;
					foreach (Member member in members.Values) if (member.MVP) mvpCount++;
					//only worry about getting the rest if there are members
					lblMemberCount.Text=memberCount.ToString()+" top members ("+mvpCount+" MVP) from the Who's Who ("+
						CP.ReputationPageCount+"R+"+CP.ArticlesPageCount+"A+"+CP.MessagesPageCount+"M)";
					dgvMembers.Columns.Clear();
					//dgvMembers.DataSource=dtMembers;
					dgvMembers.DataSource=dgvMembersBindingSource=new BindingSource(dtMembers, null);
					dgvMembers.Columns.RemoveAt(DGVM_COL_IMAGE);
					dgvMembers.Columns.Insert(DGVM_COL_IMAGE, new DataGridViewImageColumn());
					dgvMembers.Columns[DGVM_COL_IMAGE].DataPropertyName="MVP";
					dgvMembers.Columns[DGVM_COL_IMAGE].SortMode=DataGridViewColumnSortMode.NotSortable;
					dgvMembers.Columns[DGVM_COL_IMAGE].Name="MVP";
					dgvMembers.Columns.RemoveAt(DGVM_COL_NAME);
					dgvMembers.Columns.Insert(DGVM_COL_NAME, new DataGridViewLinkColumn());
					dgvMembers.Columns[DGVM_COL_NAME].DataPropertyName="Name";
					dgvMembers.Columns[DGVM_COL_NAME].SortMode=DataGridViewColumnSortMode.Automatic;
					dgvMembers.Columns[DGVM_COL_NAME].Name="Name";
					adjustMemberColumns();
					dgvMembers.Visible=true;
					btnShift.Visible=true;
					cbBlink.Visible=true;
					dgvMembers.Sort(dgvMembers.Columns["Total"], ListSortDirection.Descending);	// sort by Total
				} else {//there are no members to show, so update GUI to show this
					dgvMembers.Visible=false;
				}
			}
			lblProgress.Text="";
			prgBar.Visible=false;
			btnTopMembers.Enabled=true;
			Cursor=Cursors.Default;
		}

		private void dgvMembers_CellContentClick(object sender, DataGridViewCellEventArgs e) {
			try {
				log("dgvMembers_CellContentClick: row="+e.RowIndex+"  col="+e.ColumnIndex);
				if (e.RowIndex>=0 && e.ColumnIndex==DGVM_COL_NAME) {// name acts as a hyperlink to personal page
					DataGridViewRow row=dgvMembers.Rows[e.RowIndex];
					string rowName=row.Cells["Name"].Value.ToString();
					log("rowName="+rowName);
					Member member=members[rowName];
					log("member="+member.Name+"  #"+member.MemberID);
					clickedMember=member;
					if (member!=null && member.MemberID!=0) {
						if ((Control.ModifierKeys&Keys.Control)==0) {
							dgvMembers.Refresh();
							launchWebPage(CPSite.GetMemberPageUrl(member.MemberID));
						} else {
							tabControl2.SelectedTab=tpOneMember;
							tbMember.Text=member.MemberID.ToString();
							btnOneMember.PerformClick();
						}
					}
				}
			} catch (Exception exc) {
				// there is an occasional exception here, an out-of-bound index
				// still not sure how it occurs
				logSilent(exc);
			}
		}

		private void launchWebPage(string url) {
			try {
				Cursor=Cursors.WaitCursor;
				log("Calling web page: "+url);
				Process.Start(url);
			} catch (Exception exc) {
				logWithMessageBox(exc);
			} finally {
				Cursor=Cursors.Default;
			}
		}

		private void dgvMembers_CellFormatting(object sender, DataGridViewCellFormattingEventArgs e) {
			if (!dgvMembers.Visible) return;	// the Name column does not always exist!
			if (e.RowIndex<0) return;
			if (e.ColumnIndex>=DGVM_COL_PLATINUMCOUNT) e.CellStyle.Font=numberFont;
			DataGridViewRow row=dgvMembers.Rows[e.RowIndex];
			if (e.ColumnIndex>=DGVM_COL_PLATINUMCOUNT && (int)row.Cells[e.ColumnIndex].Value==0) return;
			string rowName=row.Cells["Name"].Value.ToString();
			Member member=members[rowName];
			if (member!=null) {
				Color backColor=Color.White;
				Color maxColor=Color.Red;
				switch (e.ColumnIndex) {
					case DGVM_COL_NUMBER:
						if (rowName==name) backColor=Color.Lime;
						break;
					case DGVM_COL_NAME:
						//if (blinkNowShows && rowName==name) backColor=Color.Lime;
						break;
					case DGVM_COL_REP_AUTHOR:
					case DGVM_COL_REP_AUTHORITY:
					case DGVM_COL_REP_DEBATOR:
					case DGVM_COL_REP_EDITOR:
					case DGVM_COL_REP_ENQUIRER:
					case DGVM_COL_REP_ORGANISER:
					case DGVM_COL_REP_PARTICIPANT:
					case DGVM_COL_REP_TOTAL:
						backColor=member.GetReputationComponentColor(e.ColumnIndex-DGVM_COL_FIRSTREP);
						if (blinkNowShows) {
							int max=champion.GetReputationComponentValue(e.ColumnIndex-DGVM_COL_FIRSTREP);
							if (max!=0 && max==member.GetReputationComponentValue(e.ColumnIndex-DGVM_COL_FIRSTREP)) {
								backColor=maxColor;
							}
						}
						break;
					case DGVM_COL_ARTICLES:
					case DGVM_COL_TIPS:
					case DGVM_COL_BLOGS:
					case DGVM_COL_MESSAGES:
					case DGVM_COL_QUESTIONS:
					case DGVM_COL_ANSWERS:
					case DGVM_COL_COMMENTS:
						if (blinkNowShows) {
							int max=champion.GetCounterValue(e.ColumnIndex-DGVM_COL_FIRSTCOUNTER);
							if (max!=0 && max==member.GetCounterValue(e.ColumnIndex-DGVM_COL_FIRSTCOUNTER)) {
								backColor=maxColor;
							}
						}
						break;
				}
				e.CellStyle.BackColor=backColor;
			}
		}

		private void dgvArticles_Sorted(object sender, EventArgs e) {
			// number the rows in the sort order
			SortOrder sortOrder=dgvArticles.SortOrder;
			if (dgvArticles.SortedColumn.Index>=3) {
				// inverse numbering for counts
				if (sortOrder==SortOrder.Descending) sortOrder=SortOrder.Ascending;
				else if (sortOrder==SortOrder.Ascending) sortOrder=SortOrder.Descending;
			}
			switch (sortOrder) {
				case SortOrder.Ascending:
					int i=0;
					foreach (DataGridViewRow row in dgvArticles.Rows) {
						if (row.Cells["Title"].Value.ToString()!="") row.Cells["#"].Value=(++i).ToString();
					}
					break;
				case SortOrder.Descending:
					i=dgvArticles.Rows.Count;
					foreach (DataGridViewRow row in dgvArticles.Rows) {
						if (row.Cells["Title"].Value.ToString()!="") row.Cells["#"].Value=(--i).ToString();
					}
					break;
				case SortOrder.None:
					foreach (DataGridViewRow row in dgvArticles.Rows) {
						if (row.Cells["Title"].Value.ToString()!="") row.Cells["#"].Value="";
					}
					break;
			}
			foreach (DataGridViewRow row in dgvArticles.Rows) {
				string title=row.Cells["Title"].Value.ToString();
				string tip="";
				if (title.Length!=0) {
					//log("title="+title);
					DateTime updated=(DateTime)row.Cells["Updated"].Value;
					string upd=updated.ToString("yyyyMMMdd.HHmmss");
					string url=titlesToUrls[title+upd];
					tip=title+Environment.NewLine+url;
				}
				row.Cells["Title"].ToolTipText=tip;
				if (totalViews!=0) {
					double percent=100.0*(int)row.Cells["Views"].Value/totalViews;
					if (percent>0) row.Cells["Views"].ToolTipText=percent.ToString("F1")+"%";
				}
				if (totalVotes!=0) {
					double percent=100.0*(int)row.Cells["Votes"].Value/totalVotes;
					if (percent>0) row.Cells["Votes"].ToolTipText=percent.ToString("F1")+"%";
				}
				if (totalBookmarks!=0) {
					double percent=100.0*(int)row.Cells["Bookmarks"].Value/totalBookmarks;
					if (percent>0) row.Cells["Bookmarks"].ToolTipText=percent.ToString("F1")+"%";
				}
				if (totalDownloads!=0) {
					double percent=100.0*(int)row.Cells["Downloads"].Value/totalDownloads;
					if (percent>0) row.Cells["Downloads"].ToolTipText=percent.ToString("F1")+"%";
				}
			}
		}

		private void setMembersTitle(string title) {
			membersTitle=title;
			Text="CP Vanity -- "+membersTitle;
		}

		private void dgvMembers_Sorted(object sender, EventArgs e) {
			clickedMember=null;
			SortOrder sortOrder=dgvMembers.SortOrder;
			if (dgvMembers.SortedColumn==dgvMembers.Columns[DGVM_COL_NAME]) {
				// inverse numbering for names
				if (sortOrder==SortOrder.Descending) sortOrder=SortOrder.Ascending;
				else if (sortOrder==SortOrder.Ascending) sortOrder=SortOrder.Descending;
			}
			dgvMembersBindingSource.RaiseListChangedEvents=false;
			switch (sortOrder) {
				case SortOrder.Descending:
					setMembersTitle("Top "+dgvMembers.SortedColumn.Name);
					int i=0;
					foreach (DataGridViewRow row in dgvMembers.Rows) {
						int PK=(int)row.Cells["PK"].Value;
						dtMembers.Rows[PK]["#"]=++i;
					}
					break;
				case SortOrder.Ascending:
					setMembersTitle("Bottom "+dgvMembers.SortedColumn.Name);
					i=dgvMembers.Rows.Count;
					foreach (DataGridViewRow row in dgvMembers.Rows) {
						int PK=(int)row.Cells["PK"].Value;
						dtMembers.Rows[PK]["#"]=i--;
					}
					break;
				case SortOrder.None:
					foreach (DataGridViewRow row in dgvMembers.Rows) {
						int PK=(int)row.Cells["PK"].Value;
						dtMembers.Rows[PK]["#"]=0;
					}
					break;
			}
			dgvMembersBindingSource.RaiseListChangedEvents=true;
			toggleCounterShift(false);	// restore shift state
		}

		private void pbCodeProject_Click(object sender, EventArgs e) {
			try { if (pbCodeProject.Image!=null) Process.Start(CP.BaseURL);
			} catch (Exception exc) { logSilent(exc); }
		}

		private void pbCompany_Click(object sender, EventArgs e) {
			try {if (pbCompany.Image!=null) Process.Start(COMPANY_SITE_URL);
			} catch (Exception exc) { logSilent(exc); }
		}

		private void lblProfilePage_LinkClicked(object sender, LinkLabelLinkClickedEventArgs e) {
			if (memberID!=0) launchWebPage(CPSite.GetMemberPageUrl(memberID));
		}

		private void lblFAQ_LinkClicked(object sender, LinkLabelLinkClickedEventArgs e) {
			launchWebPage(CPSite.GetReputationFaqUrl());
		}

		private void lblOneMember_Click(object sender, EventArgs e) {
			tabControl2.SelectedTab=tpOneMember;
			Text="CP Vanity";
			if (memberID!=0) Text="CP Vanity -- "+memberID.ToString();
			lblOneMember.Font=menuFontSelected;
			lblTopMembers.Font=menuFontUnselected;
			if ((Control.ModifierKeys&Keys.Control)==0 && btnOneMember.Enabled &&
				pbReputation.Image==null) getOneMember();
		}

		private void lblTopMembers_Click(object sender, EventArgs e) {
			tabControl2.SelectedTab=tpTopMembers;
			Text=membersTitle;
			lblOneMember.Font=menuFontUnselected;
			lblTopMembers.Font=menuFontSelected;
			if ((Control.ModifierKeys&Keys.Control)==0 && btnTopMembers.Enabled &&
				!dgvMembers.Visible) getTopMembers();
		}

		private void lblMenu_MouseEnter(object sender, EventArgs e) {
			Label label=sender as Label;
			label.BackColor=menuLightGreen;
			label.ForeColor=Color.Black;
		}

		private void lblMenu_MouseLeave(object sender, EventArgs e) {
			Label label=sender as Label;
			label.BackColor=menuDarkGreen;
			label.ForeColor=Color.White;
		}

		private void frmMain_KeyDown(object sender, KeyEventArgs e) {
			if (e.Control) {
				log("CTRL/"+e.KeyCode);
				switch (e.KeyCode) {
					case Keys.F5:
						if (tabControl2.SelectedTab==tpOneMember) btnOneMember.PerformClick();
						if (tabControl2.SelectedTab==tpTopMembers) btnTopMembers.PerformClick();
						break;
					case Keys.O:
						lblOneMember_Click(null, null);
						break;
					case Keys.H:
						lblTopMembers_Click(null, null);
						break;
					default:
						return;
				}
				e.SuppressKeyPress=true;
				e.Handled=true;
			}
			log("e.KeyCode="+e.KeyCode+"  e.KeyData="+e.KeyData+"  Keys.Alt="+Keys.Alt);
			if (e.KeyCode==Keys.ShiftKey) {
				log("this.tabControl2.TabIndex="+this.tabControl2.TabIndex);
				toggleCounterShift(true);
			}
		}

		private void cbBlink_CheckedChanged(object sender, EventArgs e) {
			withBlink=cbBlink.Checked;
			writeToRegistry("Blinking", withBlink?1u:0);
			blinkNowShows=false;
			blinkTimer.Enabled=withBlink;
			dgvMembers.Invalidate();
		}

		private bool seeCounters=false;
		private void dgvMembers_CellClick(object sender, DataGridViewCellEventArgs e) {
			// rep columns are sorted explicitly as we want to force descending sort order
			// when a rep column gets sorted for the first time
			int col=e.ColumnIndex;
			if (e.RowIndex<0) {
				if (col>=DGVM_COL_PLATINUMCOUNT && col<=DGVM_COL_LASTCOUNTER) {
					if (col!=dgvMembersSortColumn) dgvMembersSortOrder=ListSortDirection.Descending;
					dgvMembers.Sort(dgvMembers.Columns[e.ColumnIndex], dgvMembersSortOrder);
					if (dgvMembersSortOrder==ListSortDirection.Descending) {
						dgvMembersSortOrder=ListSortDirection.Ascending;
					} else {
						dgvMembersSortOrder=ListSortDirection.Descending;
					}
				}
				dgvMembersSortColumn=col;
				dgvMembers.FirstDisplayedScrollingRowIndex=0;
			}
		}

		private void btnShift_Click(object sender, EventArgs e) {
			toggleCounterShift(true);
		}

		private void toggleCounterShift(bool toggle) {
			log("toggle countershift");
			if (toggle) seeCounters=!seeCounters;
			if (seeCounters) {
				dgvMembers.FirstDisplayedScrollingColumnIndex=DGVM_COL_COMMENTS;
			} else {
				dgvMembers.FirstDisplayedScrollingColumnIndex=DGVM_COL_IMAGE;
			}
		}

		private void dgvMembers_CellToolTipTextNeeded(object sender, DataGridViewCellToolTipTextNeededEventArgs e) {
			if (e.RowIndex>=0) {
				log("CellToolTipTextNeeded "+e.RowIndex+" "+e.ColumnIndex);
				if (e.ColumnIndex==DGVM_COL_PLATINUMCOUNT) e.ToolTipText=null;
				if (e.ColumnIndex>=DGVM_COL_FIRSTREP && e.ColumnIndex<=DGVM_COL_LASTREP) {
					try {
						string name=dgvMembers.Rows[e.RowIndex].Cells[DGVM_COL_NAME].Value.ToString();
						Member member=members[name];
						int points=member.GetReputationComponentValue(e.ColumnIndex-DGVM_COL_FIRSTREP);
						string s="";
						if (points!=0) {
							int maxPoints=champion.GetReputationComponentValue(e.ColumnIndex-DGVM_COL_FIRSTREP);
							double percent=100.0*points/maxPoints;
							s=percent.ToString("F2")+"%";
						}
						e.ToolTipText=s;
					} catch (Exception exc) {
						logSilent(exc);
					}
				}
				if (e.ColumnIndex>=DGVM_COL_FIRSTCOUNTER && e.ColumnIndex<=DGVM_COL_LASTCOUNTER) {
					try {
						string name=dgvMembers.Rows[e.RowIndex].Cells[DGVM_COL_NAME].Value.ToString();
						Member member=members[name];
						int counter=member.GetCounterValue(e.ColumnIndex-DGVM_COL_FIRSTCOUNTER);
						string s="";
						if (counter!=0) {
							int maxPoints=champion.GetCounterValue(e.ColumnIndex-DGVM_COL_FIRSTCOUNTER);
							double percent=100.0*counter/maxPoints;
							s=percent.ToString("F2")+"%";
						}
						e.ToolTipText=s;
					} catch (Exception exc) {
						logSilent(exc);
					}
				}
			}
		}

		private void lblProgram_Click(object sender, EventArgs e) {
			try { Process.Start(CPVANITY_ARTICLE); } catch (Exception exc) { log(exc.ToString()); }
		}

		private void pbReputation_Click(object sender, EventArgs e) {
			if (bmReputation!=null) {
				RepGraphForm rgf=new RepGraphForm();
				rgf.Bitmap=bmReputation;
				rgf.ShowDialog();
			}
		}
	}
}