﻿//==============================================================================================
//
//	LPCodeRecognizer Test Bench
//	V2.2
//	by Luc Pattyn, January 2010
//
//	a test bench for LPCodeRecognizer
//
//	described in my CodeProject article:
//	http://www.codeproject.com/KB/applications/LPCodeRecognizer.aspx
//
//==============================================================================================
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Threading;
using System.Web;						// HttpUtility
using System.Windows.Forms;

using LPCodeRecognizer;

namespace LPCodeRecognizer.TestBench {
	public partial class Form1 : Form {
		private string PROGRAM="LPCodeRecognizer Test Bench";
		private string VERSION="V2.2";
		private System.Windows.Forms.Timer timer;
		private bool textChanged;
		private string text;

		public int test() {
			// test method with lots of less than, greater than, ampersand
			List<int> list=new List<int>();
			for (int i=0; i<10; i++) list.Add(i);
			while (list[0]>5 && list[0]<10) list.RemoveAt(0);
			return list.Count;
		}

		public Form1() {
			InitializeComponent();
			LPCodeRecognizer.Logger=new Action<string>(log);
			Language.Logger=new Action<string>(log);
			Untabber.Logger=new Action<string>(log);
			if ((Control.ModifierKeys&Keys.Control)==0) {
				Width=SystemInformation.PrimaryMonitorSize.Width-200;
				Height=SystemInformation.PrimaryMonitorSize.Height-100;
			} else {
				// hold Control key for fixed size (see img tag in HTML article)
				Width=900;
				Height=800;
			}
			Help();
			timer=new System.Windows.Forms.Timer();
			timer.Tick+=new EventHandler(timer_Tick);
			timer.Interval=1000;
			timer.Start();
			clear();
			Language.CreateAllLanguages();
			toolTip1.SetToolTip(rbAuto, "Use PRE tags when pasted text is recognized");
			toolTip1.SetToolTip(rbCode, "Always use PRE tags; use C# when unrecognized");
			toolTip1.SetToolTip(rbText, "Don't use PRE tags, don't inspect pasted text");
		}

		private void log(string s) {
			s=DateTime.Now.ToString("HH:mm:ss.fff  ")+s;
			lb.Items.Add(s);
			lb.TopIndex=lb.Items.Count-1;
		}

		private void btnAbout_Click(object sender, EventArgs e) {
			Help();
		}

		private void Help() {
			lb.Items.Clear();
			log("LPCodeRecognizer "+LPCodeRecognizer.Version+" and "+PROGRAM+" "+VERSION+" by Luc Pattyn, January 2010");
			log("");
			log("1. Use the keyboard to enter some text in the top left TextBox;");
			log("   then paste some text using Clipboard and CTRL/V,");
			log("   or click one of the paste buttons to paste built-in snippets.");
			log("   Pasting will be handled according to the \"Paste Mode\" radio buttons and checkbox.");
			log("2. View the result as rendered by the WebBrowser at the top right (updated once per second).");
			log("   Some CSS styles get added to somewhat mimic CodeProject forum pages.");
			log("3. Read log information in the bottom ListBox.");
			log("   The 'trust' value is in the range [0,100] and indicates how trustworthy the recognition result is.");
			log("");
			log("The language codes used are:");
			log(LanguageSQL.LanguageNames);
			log("");
		}

		private void btnClear_Click(object sender, EventArgs e) {
			clear();
		}

		private void clear() {
			log("clear");
			textBox1.TextChanged-=textBox1_TextChanged;
			text="[ Enter text or code here ]";
			textBox1.Text=text;
			textBox1.Select(0, text.Length);
			textChanged=false;
			webBrowser1.DocumentText=HtmlConverter.ToHtmlDocument("[ Preview ]");
			textBox1.TextChanged+=textBox1_TextChanged;
			textBox1.Focus();
		}

		private void textBox1_TextChanged(object sender, EventArgs e) {
			textChanged=true;
		}

		private void timer_Tick(object sender, EventArgs e) {
			if (textChanged) {
				panWebLED.BackColor=Color.Red;
				panWebLED.Refresh();
				textChanged=false;
				webBrowser1.DocumentText=HtmlConverter.ToHtmlDocument(textBox1.Text);
			}
		}

		private void textBox1_KeyDown(object sender, KeyEventArgs e) {
			if (e.Control && e.KeyCode==Keys.V) {
				log("CTRL/V");
				// replace selected text by cliptext, possibly code-processed (PRE, HTML encoded, etc)
				pasteClipboard();
				e.Handled=true;
				e.SuppressKeyPress=true;
			}
		}

		private void pasteClipboard() {
			paste(null, Clipboard.GetText());
		}

		private void paste(string title, string text) {
			int probability;
			Language language=null;
			if (!rbText.Checked) language=LPCodeRecognizer.Recognize(text, Language.AllLanguages, out probability);
			if (rbCode.Checked && language==null) language=Language.LanguageCS;
			if (language!=null) {
				if (cbReduceTabs.Checked) text=Untabber.ReduceLeftTabs(text);
				text="<pre lang='"+language.Name+"'"+">\r\n"+HttpUtility.HtmlEncode(text)+"</pre>";
			}
			if (title!=null) {
				text=Environment.NewLine+"<----------"+title+"----------<"+Environment.NewLine+
				text+Environment.NewLine+">----------"+title+"---------->"+Environment.NewLine;
			}
			textBox1.SelectedText=text;
		}

		private void btnHtml_Click(object sender, EventArgs e) {
			log("");
			log("HTML in browser:");
			foreach (string s in webBrowser1.DocumentText.Split('\n', '\r')) {
				if (s.Length!=0) log(s);
			}
			log("");
		}

		private void webBrowser1_DocumentCompleted(object sender, WebBrowserDocumentCompletedEventArgs e) {
			Thread.Sleep(50);
			panWebLED.BackColor=SystemColors.Control;
		}

		private void add(string s) {
			text+=s+Environment.NewLine;
		}

		private void btnText_Click(object sender, EventArgs e) {
			text="";
			add("- have a \"string Recognize(string text, out int trust)\" method that takes some text (not HTML encoded yet)");
			add("and tries to determine the (main) programming language by looking for keywords (mostly at left of lines)");
			add("and special characters such as angles, curlies, dollars; it calculates a score for every supported language,");
			add("chooses the highest score, and calculates a trust value in [0,100], and returns a string code for the resulting");
			add("language (or null if totally unclear).");
			add("- have a test app that takes text, accepts pasting, recognizes the language, shows it all in a WebBrowser");
			add("(without syntax coloring!) and shows reasoning in a ListBox.");
			add("- not all languages supported yet (HTML, XML, SQL, ...);");
			paste("Test text", text);
		}

		private void btnMath_Click(object sender, EventArgs e) {
			text="";
			add("			a00 * x + a01 * y + a02 * z = b0");
			add("			a10 * x + a11 * y + a12 * z = b1");
			add("			a20 * x + a21 * y + a22 * z = b2");
			paste("Math", text);
		}

		private void btnCS_Click(object sender, EventArgs e) {
			text="";
			add("	private void btnHtml_Click(object sender, EventArgs e) {");
			add("		log(\"HTML in browser:\");");
			add("		foreach (string s in webBrowser1.DocumentText.Split(CR, LF)) {");
			add("			if (s.Length!=0) log(s);");
			add("		}");
			add("		// an empty line follows:");
			add("");
			add("		log(\"done\");");
			add("	}");
			paste("Test C#", text);
		}

		private void btnCS3_Click(object sender, EventArgs e) {
			text="";
			add("		foreach (ItemType it in items.Items)");
			add("		{");
			add("			if (it is CalendarItemType)");
			add("			{");
			add("				CalendarItemType cal = (CalendarItemType)it;");
			add("				if (cal.ExtendedProperty != null)");
			add("					tmp.Add(new ExtendedAppointmentData()");
			add("					{");
			add("						EndDate = cal.End,");
			add("						StartDate = cal.Start,");
			add("						EventID = new Guid(cal.ExtendedProperty[0].Item.ToString()),");
			add("						AppointmentStatus = cal.LegacyFreeBusyStatus.ToString()");
			add("					});");
			add("			}");
			add("		}");
			paste("Test C# 3.0", text);
		}

		private void btnC_Click(object sender, EventArgs e) {
			text="";
			add("void activateChannel(unsigned char channel, unsigned int frequency) {");
			add("	unsigned int divider = 1193180 / frequency;");
			add("	outportByte(0x43, channel == 0 ? 0x36 : 0xB6);");
			add("	outportByte(0x40 + channel, (unsigned char)(divider & 0xFF));");
			add("	outportByte(0x40 + channel, (unsigned char)((divider >> 8) & 0xFF);");
			add("}");
			paste("Test C", text);
		}

		private void btnCPP_Click(object sender, EventArgs e) {
			text="";
			add("bool RippeEffect::Init() {");
			add("	srand(::GetTickCount()); //random seeds");
			add("	bih.biSize =sizeof(BITMAPINFOHEADER); ");
			add("	bih.biBitCount =24 ;");
			add("	bih.biCompression =BI_RGB;");
			add("	bih.biSizeImage=m_nBmHeight*m_nBmWidth*bih.biBitCount /8;");
			add("	bih.biXPelsPerMeter=0;");
			add("}");
			paste("Test C++", text);
		}

		private void btnCPPCLI_Click(object sender, EventArgs e) {
			text="";
			add("public ref class personalDataGridView : DataGridView {	");
			add("public:");
			add("	personalDataGridView() {");
			add("		this->selectedRowIndexes=gcnew Collections::ArrayList();");
			add("		this->MultiSelectForced=false;");
			add("	}");
			add("	property bool MultiSelectForced {");
			add("		bool get() {");
			add("			return _MultiSelectForced;");
			add("		} ");
			add("		void set(bool value) {");
			add("			this->_MultiSelectForced = value;");
			add("			this->selectedRowIndexes->Clear();");
			add("			processSelectedRowIndexes();");
			add("		}");
			add("	}");
			paste("Test C++/CLI", text);
		}

		private void btnVBNET_Click(object sender, EventArgs e) {
			text="";
			add("	Private Sub log(ByVal threadID As Integer, ByVal s As String)");
			add("		If Me.InvokeRequired Then");
            add("			Me.Invoke(logHandler, New Object() {threadID, s})");
			add("		Else");
            add("			If s.Length > 0 Then");
            add("		    Dim threadString As String = \"      \"");
            add("		    threadID = threadID And 255");
            add("		    If threadID <> 0 Then threadString = \" [\" & threadID.ToString(\"X2\") & \"] \"");
            add("			    s = DateTime.Now.ToString(\"HH:mm:ss.fff\") & threadString & s");
            add("			End If");
            add("			If lb IsNot Nothing Then lb.Items.Add(s)");
			add("		End If");
			add("	End Sub");
			paste("Test VB.NET", text);
		}

		private void btnSQL_Click(object sender, EventArgs e) {
			text="";
			add("@iParam varchar(100)  AS ");
			add("SELECT ipkClientsID AS [ID], tblClients.sClientName AS [Client: Client Name], ");
			add("(SELECT CONVERT(varchar(8000), sTextValue) FROM tblReportSummary WHERE       ifkID = tblClients.ipkClientsID AND       iParam = @iParam AND       sFieldName = 'Client: Postal Address Group (R)')  AS [Client: Postal Address Group (R)] FROM tblClients ");
			add("WHERE ipkClientsID IN (SELECT ifkSelectedItemsID from tblSelectedItems where sID = @iParam)");
			paste("Test SQL", text);
		}

		private void btnSQL2_Click(object sender, EventArgs e) {
			text="";
			add("@Text ='%hema%'");
			add("@itemcount int");
			add("SET @SQL ='SELECT COUNT(*) FROM dbo.table");
			add("WHERE (LOWER(column) LIKE LOWER('+@Text+') OR LOWER(column) LIKE LOWER('+@Text+'))");
			add("exec @SQL");
			paste("Test SQL2", text);
		}

		private void btnPHP_Click(object sender, EventArgs e) {
			text="";
			add("$DBG=$_POST['DBG'];");
			add("if ($DBG!=null) {");
			add("	$row=$DB->FindOne(\"SELECT * FROM envTable WHERE id='0'\");");
			add("	if ($row) {");
			add("		$query=\"UPDATE envTable SET debugLevel='$DBG' WHERE id='0'\";");
			add("		$DB->WriteQuery($query);");
			add("	} else {");
			add("		show(\"Failed to set debug level\");");
			add("	}");
			add("}");
			paste("Test PHP", text);
		}

		private void btnCSS_Click(object sender, EventArgs e) {
			text="";
			add("#LP p.fix {					/*	class='fix'	fixed font for code, productID, etc.	*/");
			add("	font-family:courier;");
			add("	font-size: 14px;");
			add("	font-weight:400;");
			add("}");
			add("#LP p.debug {					/*	class='debug'	magenta bold font for debug			*/");
			add("	font-weight: bold;");
			add("	font-size: 12px;");
			add("	color: #f0f;");
			add("}");
			paste("Test CSS", text);
		}

		private void btnJava_Click(object sender, EventArgs e) {
			text="";
			add("private static String loadLibrary() {");
			add("	for (int i=0; i<nativeDLLfiles.length; i++) {");
			add("		String libraryLoaded=nativeDLLfiles[i];");
			add("		try {");
			add("			logJ(\": Trying to load \"+libraryLoaded);");
			add("			System.loadLibrary(libraryLoaded); 	// without .dll extension !!");
			add("			logJ(\": Load attempt succeeded\");");
			add("			try {loadLibraryC();}");
			add("			catch (Throwable e2) {}");
			add("			return libraryLoaded;");
			add("			}");
			add("		catch (Throwable e1) {");
			add("			libraryLoaded=null;");
			add("		}");
			add("	}");
			add("	logJ(\": All load attempts failed\");");
			add("	return null;");
			add("}");
			paste("Test Java", text);
		}

		private void btnXML_Click(object sender, EventArgs e) {
			text="";
			add("<?xml version=\"1.0\" encoding=\"utf-8\" ?>");
			add("<Languages>");
			add("	<Block ID=\"C|CPP|C++|mc++\">");
			add("		<Token Value=\"(#define|#error|#import|#undef|#elif|#if|#using|#region|#endregion)\">");
			add("			<PreData>&lt;span class=&quot;code-preprocessor&quot;&gt;</PreData>");
			add("			<PostData>&lt;/span&gt;</PostData>");
			add("		</Token>");
			add("	</Block>");
			add("</Languages>");
			paste("Test XML", text);
		}

		private void btnHTML_Click_1(object sender, EventArgs e) {
			text="";
			add("<h2>Conclusion</h2>");
			add("<p>A few simple rules suffice to give a reasonable estimate as to what the main programming language is in a code snippet.");
			add("When used interactively while pasting such snippet in a message editor and automatically generating some mark-up, the user");
			add("could still modify it whenever he does not agree with the result.</p>");
			paste("Test HTML", text);
		}

		private void btnASM_Click(object sender, EventArgs e) {
			text="";
			add("_TEXT   SEGMENT");
			add("_argc$ = 8");
			add("_argv$ = 12");
			add("_main   PROC NEAR ; COMDAT");
			add("	push    OFFSET FLAT:??_C@_0N@GCDOMLDM@Hello?5World?$CB?$AA@");
			add("	call    _printf");
			add("	add esp, 4");
			add("	xor eax, eax");
			add("	ret 0");
			paste("Test ASM", text);
		}

		private void btnMSIL_Click(object sender, EventArgs e) {
			text="";
			add(".maxstack  1");
			add(".locals init (int32 V_0)");
			add("IL_0000:  ldarg.1");
			add("IL_0001:  brfalse.s  IL_0007");
			add("IL_0003:  ldc.i4.1");
			add("IL_0004:  stloc.0");
			add("IL_0005:  br.s       IL_0009");
			add("IL_0007:  ldc.i4.2");
			add("IL_0008:  stloc.0");
			add("IL_0009:  ldloc.0");
			add("IL_000a:  ret");
			paste("Test MSIL", text);
		}

		private void btnJavaScript_Click(object sender, EventArgs e) {
			text="";
			add("function DisplayString() {");
			add("	alert( this.myString ); ");
			add("}");
			add("var myClassObj1 = new MyClass();");
			add("var myClassObj2 = new MyClass();");
			add("myClassObj1.myData = 10;");
			add("myClassObj1.myString = \"Obj1:  Hello World\";");
			add("myClassObj2.myData = 20;");
			add("myClassObj2.myString = \"Obj2:  Hello World\";");
			add("myClassObj1.ShowData();     // displays: 10");
			add("myClassObj1.ShowString();   // displays: \"Obj1:  Hello World\"");
			add("myClassObj2.ShowData();     // displays: 20");
			paste("Test JavaScript", text);
		}

		private void btnVBScript_Click(object sender, EventArgs e) {
			text="";
			add("function fmt( str, args )");
			add("	dim i");
			add("	for i = 1 to Len(str)");
			add("		' found a fmt char.");
			add("		if i<Len(str) then");
			add("			i = i + 1");
			add("		end if");
			add("	next");
			add("	fmt = res");
			add("end function");
			paste("Test VBScript", text);
		}

		private void btnASPNET_Click(object sender, EventArgs e) {
			text="";
			add("<asp:EntityDataSource ID=\"detailsDataSource\" runat=\"server\" EnableDelete=\"true\" EnableInsert=\"true\" EnableUpdate=\"true\">");
			add("	<WhereParameters>");
			add("		<asp:DynamicQueryStringParameter />");
			add("	</WhereParameters>");
			add("</asp:EntityDataSource>");
			paste("Test ASP.NET", text);
		}

		private void btnMIDL_Click(object sender, EventArgs e) {
			text="";
			add("[");
			add("  uuid(ba209999-0c6c-11d2-97cf-00c04f8eea45),");
			add("  version(1.0)");
			add("]");
			add("interface MyInterface");
			add("{");
			add("  const unsigned short INT_ARRAY_LEN = 100;");
			add("  void MyRemoteProc( ");
			add("	  [in] int param1,");
			add("	  [out] int outArray[INT_ARRAY_LEN]");
			add("  );");
			add("}");
			paste("Test MIDL", text);
		}

#if false
#endif
	}
}
