﻿//==============================================================================================
//
//	Language
//	V2.0
//	by Luc Pattyn, January 2010
//
//	base class used by LPCodeRecognizer
//
//==============================================================================================
using System;
using System.Collections.Generic;
using System.Text;

#if false
		C-like:	C#, C, C++, C++.CLI, Java, JavaScript
		XML-like: XML, HTML, ASP, ASP.NET
		VB-like: VB, VBScript
		F#
		PHP
		CSS
		SQL
		ASM
		MSIL
		MIDL
#endif

namespace LPCodeRecognizer {

	public class Language {
		public static Action<string> Logger;
		public static List<Language> AllLanguages;
		public static Language LanguageCS;
		public string Name;
		public string AlternateNames;
		public List<string> Keywords;
		public string[] Checks;
		public bool CaseSensitive;	// keyword case matters
		public int LhsRecognize;	// number of lefthand side words that should be keywords
		public List<string> Hotspots;
		public List<string> Coldspots;

		// (input) string name: the name that will be returned by the Name property; e.g. "cs" for Csharp.
		// (input) bool caseSensitive: true iff keywords are case sensitive (true for C#, false for VB.NET).
		// (input) int lhsRecognize: number of words on the lefthand side that should be keywords (unless
		//			followed by a special symbol such as = or (
		//			typically 2 for typed languages such as C, 1 for assembly, 0 for formatfree stuff (html)
		// (input) string checks: list of codewords describing the recognition criteria, separated by '|';
		//			the acceptable values are the cases of a single switch inside CodeRecognizer.Recognize
		// (input) string hotspots: list of special character sequences that are looked for everywhere, in a
		//			position-independent and case-insensitive way
		// warning: the actual keywords must be added by calling the add() method once or more.
		// remark: each language definition added to the collection makes it harder for CodeRecognizer to
		//			tell them apart; the trust value may go down if a language gets added that ressembles an
		//			older one.
		public Language(string name, string alternateNames, bool caseSensitive, int lhsRecognize, 
			string checks, string hotspots, string coldspots) {
			Name=name;
			AlternateNames=alternateNames;
			CaseSensitive=caseSensitive;
			LhsRecognize=lhsRecognize;
			if (!caseSensitive) hotspots=hotspots.ToLower();
			Keywords=new List<string>();
			Hotspots=new List<string>(hotspots.Split('|'));
			Coldspots=new List<string>(coldspots.Split('|'));
			Checks=checks.Split('|');
		}

		// log a message
		private void log(string s) {
			if (Logger!=null) Logger(s);
		}

		// add some keywords (a string, without white space, all keywords separated by '|')
		public void Add(string keywords) {
			if (!CaseSensitive) keywords=keywords.ToLower();
			foreach (string kw1 in keywords.Split('|')) {
				string kw=kw1.Trim();
				if (kw.Length==0) {
					log("error in keywords: "+keywords); 
				} else {
					if (!Keywords.Contains(kw)) Keywords.Add(kw);
				}
			}
		}

		public static void CreateAllLanguages() {
			AllLanguages=new List<Language>();
			AllLanguages.Add(LanguageCS=new LanguageGroupC());
			AllLanguages.Add(new LanguageGroupVB());
			AllLanguages.Add(new LanguageGroupXML());
			AllLanguages.Add(new LanguageSQL());
			AllLanguages.Add(new LanguageMIDL());
			AllLanguages.Add(new LanguagePHP());
			AllLanguages.Add(new LanguageCSS());
			AllLanguages.Add(new LanguageMSIL());
			AllLanguages.Add(new LanguageASM());
		}

		public static string LanguageNames {
			get {
				if (AllLanguages==null) CreateAllLanguages();
				string s="";
				string sep="";
				foreach (Language L in AllLanguages) {
					s=s+sep+L.Name;
					string alt=L.AlternateNames;
					if (alt.Length!=0) {
						s+="["+alt+"]";
						sep="; ";
					}
				}
				return s;
			}
		}
	}
}
