//*******************************************************************
// LP Sokoban
//
// my implementation of the popular Sokoban puzzle
// Copyright: Luc Pattyn, January 2007
//
// This code is freely available for any non-commercial use.
//
//*******************************************************************

using System;
using System.Collections;			// ArrayList
using System.Drawing;
using System.Drawing.Printing;		// PrinterSettings
using System.Windows.Forms;
using System.ComponentModel;

namespace Sokoban {
	/// <summary>
	/// Board is the board used to draw the Sokoban puzzle.
	/// </summary>
	public class Board : Panel {
		private Logger log;
		private Sokoban sokoban;
		private bool sokobanVisible;
		private Timer blinkTimer;

		public Board(Form form, Logger log, Sokoban sokoban) {
			// set 3 bits to get double buffering
			SetStyle(System.Windows.Forms.ControlStyles.DoubleBuffer,true);
			SetStyle(System.Windows.Forms.ControlStyles.AllPaintingInWmPaint,true);
			SetStyle(System.Windows.Forms.ControlStyles.UserPaint,true);
			// store parameters
			this.log=log;
			this.sokoban=sokoban;
			// create timer for blinking
			blinkTimer=new Timer();
			blinkTimer.Interval=300;
			blinkTimer.Tick+=new EventHandler(timer_Tick);
			blinkTimer.Start();
			Paint+=new PaintEventHandler(Board_Paint);
		}

		public void Repaint() {
			// set blinker to visible for better image when moving fast (e.g.animation)
			sokobanVisible=true;
			Invalidate();	// causes a repaint
		}

		// makes the Sokoban blink
		private void timer_Tick(object sender, EventArgs e) {
			sokobanVisible=!sokobanVisible || sokoban.Solved;	// full on when done
			Invalidate();
		}

		private void Board_Paint(object sender, System.Windows.Forms.PaintEventArgs e) {
			Painter(e.Graphics, Width, Height, 100, false);
		}

		// The painter method is used for screen display (in this panel) as well as
		// for printing; but printing does not (always) set useful ClipBounds,
		// so we need "panel" size parameters
		// to take care of printer margin problems, we use a fillPercentage<=100
		// (the method centers the drawing anyway)
		// bool printing is used for logging; it also forces the sokoban to be visible.
		public void Painter(Graphics g,  int panelWidth, int panelHeight, 
			int fillPercentage, bool printing) {
			if (printing || Control.MouseButtons==MouseButtons.Right) {
				log("panelSize="+panelWidth+"*"+panelHeight);
				log("clipBounds="+g.ClipBounds.ToString());
				log("dpi="+g.DpiX);
				log("pageScale="+g.PageScale);
				log("pageUnit="+g.PageUnit);
			}
			// some d variables to experiment with different sizes...
			int w=sokoban.LevelWidth;
			int h=sokoban.LevelHeight;
			if (w<1) w=1;
			if (h<1) h=1;
			// try to use the indicated percentage of the available dimension
			int dHor=panelWidth*fillPercentage/w/100;		// max hor pixels per cell
			int dVert=panelHeight*fillPercentage/h/100;		// max vert pixels per cell
			int d=dHor<dVert?dHor:dVert;		// take the smallest one
			int dd=d*9/10;
			int dg=d/10;
			int dp=d/5;							// box should be smaller than goal
			int dp2=d/3;
			int penWidth=d/20;
			if (penWidth<1) penWidth=1;
			Pen redPen=new Pen(Color.Red, penWidth);
			Pen greenPen=new Pen(Color.LimeGreen, penWidth);
			// move origin so game sits in the middle
			g.TranslateTransform((panelWidth-w*d)/2, (panelHeight-h*d)/2);
			foreach(Item item in sokoban.Walls) {
				g.FillRectangle(Brushes.Brown, d*item.x, d*item.y, dd, dd);
			}
			foreach(Item item in sokoban.Goals) {
				g.FillRectangle(Brushes.Pink, d*item.x+dg, d*item.y+dg, dd-2*dg, dd-2*dg);
			}
			foreach(Item item in sokoban.Boxes) {
				int x=d*item.x;
				int y=d*item.y;
				g.FillRectangle(Brushes.Yellow, x+dp, y+dp, dd-2*dp, dd-2*dp);
				// draw green V or red X
				if (sokoban.IsOnGoal(item.x, item.y)) {
					g.DrawLine(greenPen, x+dp2, y+dd/2, x+dd/2, y+dd-dp2);
					g.DrawLine(greenPen, x+dd-dp2, y+dp2, x+dd/2, y+dd-dp2);
				} else {
					g.DrawLine(redPen, x+dp2, y+dp2, x+dd-dp2, y+dd-dp2);
					g.DrawLine(redPen, x+dp2, y+dd-dp2, x+dd-dp2, y+dp2);
				}
			}
			if (sokobanVisible || printing) {
				// draw a filled triangle pointing in the direction of the last move
				int lastDx=sokoban.LastDx;
				int lastDy=sokoban.LastDy;
				Item me=sokoban.Me;
				int x=d*me.x+d/2;
				int y=d*me.y+d/2;
				int dds=d/3;
				int x1=lastDx*dds+x;
				int y1=lastDy*dds+y;
				int x2=(-lastDx-lastDy)*dds+x;
				int y2=(lastDx-lastDy)*dds+y;
				int x3=(-lastDx+lastDy)*dds+x;
				int y3=(-lastDx-lastDy)*dds+y;
				System.Drawing.Drawing2D.GraphicsPath path=new System.Drawing.Drawing2D.GraphicsPath();
				path.AddLine(x1, y1, x2, y2);
				path.AddLine(x2, y2, x3, y3);
				path.AddLine(x3, y3, x1, y1);
				g.FillPath(Brushes.Red, path);
			}
		}

#region board's printing stuff
		protected static PrinterSettings printerSettings=new PrinterSettings();

		public void Print() {
			try {
				PrintDocument printDoc=new PrintableBoard(this);
				PrinterSettings ps=printDialog(printDoc);
				if (ps!=null) {
					ps.DefaultPageSettings.Landscape=false;
					ps.Duplex=Duplex.Simplex;
					printDoc.PrinterSettings=ps;
					log("Printing document "+printDoc.DocumentName);
					printDoc.Print();
				}
			} catch(Exception e) {
				log(e.ToString());
			}
		}

		protected PrinterSettings printDialog(PrintDocument printDoc) {
			PrintDialog pd=new PrintDialog();
			pd.Document=printDoc;
			// start with previous settings !
			pd.PrinterSettings=printerSettings;
			PrinterSettings ps=printerSettings;
			pd.AllowPrintToFile=true;
			pd.AllowSelection=false;
			printDoc.DefaultPageSettings.Landscape=false;
			ps.DefaultPageSettings.Landscape=false;
			DialogResult result=pd.ShowDialog();
			ps=pd.PrinterSettings;
			log("printer name="+ps.PrinterName);
			log("color="+ps.SupportsColor);
			if (result!=DialogResult.OK) return null;
			return ps;
		}
#endregion
	}

	public class PrintableBoard : PrintDocument {
		private Board board;

		public PrintableBoard(Board board) {
			this.board=board;
			PrintPage+=new PrintPageEventHandler(PrintableBoard_PrintPage);
			// get form title and turn it into a valid file name
			// (in case we save to file, e.g. with some PDF writer)
			string s=board.TopLevelControl.Text.Replace(" ","_");
			DocumentName=s.Replace("-","_").Replace("#","_").Replace(".","_");
		}

		private void PrintableBoard_PrintPage(object sender, PrintPageEventArgs e) {
			Graphics g=e.Graphics;
			int w=e.PageBounds.Width;
			int h=e.PageBounds.Height;
			// draw a header line
			g.DrawString(DocumentName, new Font("Arial", 12), Brushes.Black, w/20, h/20);
			// now paint the board in the center of the page, fitting 85% of the page
			board.Painter(g, w, h, 85, true);
			throw new ApplicationException("LP");
		}
	}
}
