//*******************************************************************
// LP Sokoban
//
// my implementation of the popular Sokoban puzzle
// Copyright: Luc Pattyn, January 2007
//
// This code is freely available for any non-commercial use.
//
//*******************************************************************

using System;
using System.Collections;

namespace Sokoban {

	//####################################################################################
	/// <summary>
	/// History stores any number of objects (describing an atomic action)
	/// and provides the logic needed for undo/redo functionality.
	/// </summary>
	/// <remarks>Inspired by
	/// ref: An Undo/Redo Buffer Framework, by Marc Clifton
	/// http://www.codeproject.com/csharp/undoredobuffer.asp
	/// but trimmed to just serve the needs of the Sokoban game</remarks>
	//####################################################################################
	public class History {
		//====================================================================================
		// object data
		//====================================================================================
		private Logger log;
		private ArrayList buffer=new ArrayList();
		private int idx=0;

		//====================================================================================
		// structor
		//====================================================================================
		/// <summary>
		/// Creates an instance of History.
		/// </summary>
		public History(Logger log) {
			this.log=log;
		}

		//====================================================================================
		// public properties
		//====================================================================================
		/// <summary>
		/// Clears the History memory completely.
		/// </summary>
		public void Clear() {
			buffer.Clear();
			idx=0;
			log("History.Clear: idx="+idx);
		}

		/// <summary>
		/// Returns the current index position in the history buffer.
		/// </summary>
		public int Index {get {return idx;}}

		/// <summary>
		/// Returns true if there are items in the history buffer.
		/// </summary>
		public bool CanUndo {get {return idx>0;}}

		/// <summary>
		/// Returns true if the current position in the undo buffer will
		/// allow for redo's.
		/// </summary>
		public bool CanRedo {get {return buffer.Count > idx;}}

		//====================================================================================
		// public methods
		//====================================================================================
		/// <summary>
		/// Save a new history item.  Anything past the index position is lost (no more redo
		/// possible).
		/// </summary>
		/// <param name="history">The history object that lead to the current state.</param>
		public void Do(object history) {
			if (buffer.Count>idx) {
				log("History.Do: removing from "+idx+" to "+buffer.Count);
				buffer.RemoveRange(idx, buffer.Count-idx);
			}
			buffer.Add(history);
			++idx;
			log("History.Do: idx="+idx+": "+history.ToString());
		}

		/// <summary>
		/// Returns the current history object.
		/// </summary>
		public object Undo() {
			if (!CanUndo) throw new ApplicationException("There is nothing to undo.");
			object history=buffer[--idx];
			log("History.Undo: idx="+idx+": "+history.ToString());
			return history;
		}

		/// <summary>
		/// Returns the next history object.
		/// </summary>
		public object Redo() {
			if (!CanRedo) throw new ApplicationException("There is nothing to redo.");
			object history=buffer[idx++];
			log("History.Redo: idx="+idx+": "+history.ToString());
			return history;
		}
	}
}

