//*******************************************************************
// LP Sokoban
//
// my implementation of the popular Sokoban puzzle
// Copyright: Luc Pattyn, January 2007
//
// This code is freely available for any non-commercial use.
//
//*******************************************************************

using System;
using System.Collections;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.IO;						// Path
using System.Runtime.InteropServices;	// DllImport (MessageBeep)
using System.Threading;					// Thread
using System.Windows.Forms;

using Microsoft.Win32;					// RegistryKey


namespace Sokoban {
	/// <summary>
	/// Delegate to process a log message.
	/// </summary>
	public delegate void Logger(string message);

	/// <summary>
	/// The main form interacts with the user.
	/// </summary>
	public class MainForm : System.Windows.Forms.Form {
#region dataFields
		//=================================================================
		// some constants
		//=================================================================
		private static string PROGRAM_NAME="LP Sokoban#";

		// registry names
		private static string REGISTRY_ROOT=@"Software\LP\Sokoban";
		private static string REGISTRY_DEBUGFLAG="DebugFlag";
		private static string REGISTRY_CURRENTFILE="File";
		private static string REGISTRY_CURRENTLEVEL="Level";
		private static string REGISTRY_FORMWIDTH="FormWidth";
		private static string REGISTRY_FORMHEIGHT="FormHeight";

		//=================================================================
		// global variables
		//=================================================================
		private Sokoban sokoban;		// the game logic
		private Board board;			// a panel showing the game
		private string fileSpec=null;	// the game's source file
		private int level=0;			// the game level
		private string pasteString;		// string to paste in separate thread
		private bool pasteAbort;		// got an ESC key

		//=================================================================
		// Designer-created variables
		//=================================================================
		/// <summary>
		/// Required designer variable.
		/// </summary>
		private System.ComponentModel.Container components = null;
		
		private System.Windows.Forms.ListBox lb;
		private System.Windows.Forms.Panel prelimBoard;
		private System.Windows.Forms.Label lblStatus;

		private System.Windows.Forms.MainMenu mainMenu1;

		private System.Windows.Forms.MenuItem menuFile;
		private System.Windows.Forms.MenuItem menuOpen;
		private System.Windows.Forms.MenuItem menuRestartLevel;
		private System.Windows.Forms.MenuItem menuNextLevel;
		private System.Windows.Forms.MenuItem menuPreviousLevel;
		private System.Windows.Forms.MenuItem menuQuit;
		private System.Windows.Forms.MenuItem menuEdit;
		private System.Windows.Forms.MenuItem menuUndo;
		private System.Windows.Forms.MenuItem menuRedo;
		private System.Windows.Forms.MenuItem menuView;
		private System.Windows.Forms.MenuItem menuDebug;
		private System.Windows.Forms.MenuItem menuHelp;
		private System.Windows.Forms.MenuItem menuRules;
		private System.Windows.Forms.MenuItem menuAbout;

		private System.Windows.Forms.MenuItem menuItem5;
		private System.Windows.Forms.Label lblLevel;
		private System.Windows.Forms.MenuItem menuItem1;
		private System.Windows.Forms.MenuItem menuCopy;
		private System.Windows.Forms.MenuItem menuPaste;
		private System.Windows.Forms.MenuItem menuItem2;
		private System.Windows.Forms.MenuItem menuPrint;
		private System.Windows.Forms.MenuItem menuItem6;
#endregion

#region structor stuff
		//=================================================================
		// constructor
		//=================================================================
		public MainForm() {
			// Required for Windows Form Designer support
			InitializeComponent();

			// create a puzzle engine
			sokoban=new Sokoban(new Logger(log));

			// create a board panel and replace preliminary panel by it
			board=new Board(this, new Logger(log), sokoban);
			board.Bounds=prelimBoard.Bounds;
			Controls.Remove(prelimBoard);
			Controls.Add(board);

			// organize some event handlers
			this.KeyPreview=true;	// pass keys to form before they could go to a control
			KeyDown+=new KeyEventHandler(MainForm_KeyDown);
			Resize+=new EventHandler(MainForm_Resize);
			Closed+=new EventHandler(MainForm_Closed);
			Load+=new EventHandler(MainForm_Load);
		}

		/// <summary>
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing ) {
			if( disposing ) {
				if (components != null) {
					components.Dispose();
				}
			}
			base.Dispose( disposing );
		}

		#region Windows Form Designer generated code
		/// <summary>
		/// Required method for Designer support - do not modify
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent() {
			this.mainMenu1 = new System.Windows.Forms.MainMenu();
			this.menuFile = new System.Windows.Forms.MenuItem();
			this.menuOpen = new System.Windows.Forms.MenuItem();
			this.menuItem6 = new System.Windows.Forms.MenuItem();
			this.menuRestartLevel = new System.Windows.Forms.MenuItem();
			this.menuNextLevel = new System.Windows.Forms.MenuItem();
			this.menuPreviousLevel = new System.Windows.Forms.MenuItem();
			this.menuItem5 = new System.Windows.Forms.MenuItem();
			this.menuPrint = new System.Windows.Forms.MenuItem();
			this.menuItem2 = new System.Windows.Forms.MenuItem();
			this.menuQuit = new System.Windows.Forms.MenuItem();
			this.menuEdit = new System.Windows.Forms.MenuItem();
			this.menuUndo = new System.Windows.Forms.MenuItem();
			this.menuRedo = new System.Windows.Forms.MenuItem();
			this.menuItem1 = new System.Windows.Forms.MenuItem();
			this.menuCopy = new System.Windows.Forms.MenuItem();
			this.menuPaste = new System.Windows.Forms.MenuItem();
			this.menuView = new System.Windows.Forms.MenuItem();
			this.menuDebug = new System.Windows.Forms.MenuItem();
			this.menuHelp = new System.Windows.Forms.MenuItem();
			this.menuRules = new System.Windows.Forms.MenuItem();
			this.menuAbout = new System.Windows.Forms.MenuItem();
			this.lb = new System.Windows.Forms.ListBox();
			this.lblStatus = new System.Windows.Forms.Label();
			this.prelimBoard = new System.Windows.Forms.Panel();
			this.lblLevel = new System.Windows.Forms.Label();
			this.SuspendLayout();
			// 
			// mainMenu1
			// 
			this.mainMenu1.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																					  this.menuFile,
																					  this.menuEdit,
																					  this.menuView,
																					  this.menuHelp});
			// 
			// menuFile
			// 
			this.menuFile.Index = 0;
			this.menuFile.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																					 this.menuOpen,
																					 this.menuItem6,
																					 this.menuRestartLevel,
																					 this.menuNextLevel,
																					 this.menuPreviousLevel,
																					 this.menuItem5,
																					 this.menuPrint,
																					 this.menuItem2,
																					 this.menuQuit});
			this.menuFile.Text = "File";
			// 
			// menuOpen
			// 
			this.menuOpen.Index = 0;
			this.menuOpen.Shortcut = System.Windows.Forms.Shortcut.CtrlO;
			this.menuOpen.Text = "Open...";
			this.menuOpen.Click += new System.EventHandler(this.menuOpen_Click);
			// 
			// menuItem6
			// 
			this.menuItem6.Index = 1;
			this.menuItem6.Text = "-";
			// 
			// menuRestartLevel
			// 
			this.menuRestartLevel.Index = 2;
			this.menuRestartLevel.Shortcut = System.Windows.Forms.Shortcut.CtrlR;
			this.menuRestartLevel.Text = "Restart Level";
			this.menuRestartLevel.Click += new System.EventHandler(this.menuRestart_Click);
			// 
			// menuNextLevel
			// 
			this.menuNextLevel.Index = 3;
			this.menuNextLevel.Shortcut = System.Windows.Forms.Shortcut.CtrlF;
			this.menuNextLevel.Text = "Next Level";
			this.menuNextLevel.Click += new System.EventHandler(this.menuNextLevel_Click);
			// 
			// menuPreviousLevel
			// 
			this.menuPreviousLevel.Index = 4;
			this.menuPreviousLevel.Shortcut = System.Windows.Forms.Shortcut.CtrlB;
			this.menuPreviousLevel.Text = "Previous Level";
			this.menuPreviousLevel.Click += new System.EventHandler(this.menuPreviousLevel_Click);
			// 
			// menuItem5
			// 
			this.menuItem5.Index = 5;
			this.menuItem5.Text = "-";
			// 
			// menuPrint
			// 
			this.menuPrint.Index = 6;
			this.menuPrint.Shortcut = System.Windows.Forms.Shortcut.CtrlP;
			this.menuPrint.Text = "Print...";
			this.menuPrint.Click += new System.EventHandler(this.menuPrint_Click);
			// 
			// menuItem2
			// 
			this.menuItem2.Index = 7;
			this.menuItem2.Text = "-";
			// 
			// menuQuit
			// 
			this.menuQuit.Index = 8;
			this.menuQuit.Text = "Quit";
			this.menuQuit.Click += new System.EventHandler(this.menuQuit_Click);
			// 
			// menuEdit
			// 
			this.menuEdit.Index = 1;
			this.menuEdit.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																					 this.menuUndo,
																					 this.menuRedo,
																					 this.menuItem1,
																					 this.menuCopy,
																					 this.menuPaste});
			this.menuEdit.Text = "Edit";
			// 
			// menuUndo
			// 
			this.menuUndo.Index = 0;
			this.menuUndo.Shortcut = System.Windows.Forms.Shortcut.CtrlZ;
			this.menuUndo.Text = "Undo";
			this.menuUndo.Click += new System.EventHandler(this.menuUndo_Click);
			// 
			// menuRedo
			// 
			this.menuRedo.Index = 1;
			this.menuRedo.Shortcut = System.Windows.Forms.Shortcut.CtrlY;
			this.menuRedo.Text = "Redo";
			this.menuRedo.Click += new System.EventHandler(this.menuRedo_Click);
			// 
			// menuItem1
			// 
			this.menuItem1.Index = 2;
			this.menuItem1.Text = "-";
			// 
			// menuCopy
			// 
			this.menuCopy.Index = 3;
			this.menuCopy.Shortcut = System.Windows.Forms.Shortcut.CtrlC;
			this.menuCopy.Text = "Copy";
			this.menuCopy.Click += new System.EventHandler(this.menuCopy_Click);
			// 
			// menuPaste
			// 
			this.menuPaste.Index = 4;
			this.menuPaste.Shortcut = System.Windows.Forms.Shortcut.CtrlV;
			this.menuPaste.Text = "Paste";
			this.menuPaste.Click += new System.EventHandler(this.menuPaste_Click);
			// 
			// menuView
			// 
			this.menuView.Index = 2;
			this.menuView.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																					 this.menuDebug});
			this.menuView.Text = "View";
			// 
			// menuDebug
			// 
			this.menuDebug.Index = 0;
			this.menuDebug.Shortcut = System.Windows.Forms.Shortcut.CtrlD;
			this.menuDebug.Text = "Debug";
			this.menuDebug.Click += new System.EventHandler(this.menuDebug_Click);
			// 
			// menuHelp
			// 
			this.menuHelp.Index = 3;
			this.menuHelp.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																					 this.menuRules,
																					 this.menuAbout});
			this.menuHelp.Text = "Help";
			// 
			// menuRules
			// 
			this.menuRules.Index = 0;
			this.menuRules.Text = "Rules...";
			this.menuRules.Click += new System.EventHandler(this.menuRules_Click);
			// 
			// menuAbout
			// 
			this.menuAbout.Index = 1;
			this.menuAbout.Text = "About...";
			this.menuAbout.Click += new System.EventHandler(this.menuAbout_Click);
			// 
			// lb
			// 
			this.lb.Font = new System.Drawing.Font("Courier New", 10.13333F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.lb.HorizontalScrollbar = true;
			this.lb.ItemHeight = 21;
			this.lb.Location = new System.Drawing.Point(448, 48);
			this.lb.Name = "lb";
			this.lb.Size = new System.Drawing.Size(96, 340);
			this.lb.TabIndex = 1;
			// 
			// lblStatus
			// 
			this.lblStatus.Font = new System.Drawing.Font("Microsoft Sans Serif", 11.2F, System.Drawing.FontStyle.Bold, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.lblStatus.Location = new System.Drawing.Point(328, 8);
			this.lblStatus.Name = "lblStatus";
			this.lblStatus.Size = new System.Drawing.Size(744, 32);
			this.lblStatus.TabIndex = 2;
			this.lblStatus.Text = "label1";
			// 
			// prelimBoard
			// 
			this.prelimBoard.BackColor = System.Drawing.Color.White;
			this.prelimBoard.Location = new System.Drawing.Point(32, 48);
			this.prelimBoard.Name = "prelimBoard";
			this.prelimBoard.Size = new System.Drawing.Size(368, 368);
			this.prelimBoard.TabIndex = 0;
			// 
			// lblLevel
			// 
			this.lblLevel.Font = new System.Drawing.Font("Microsoft Sans Serif", 11.2F, System.Drawing.FontStyle.Bold, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.lblLevel.Location = new System.Drawing.Point(32, 8);
			this.lblLevel.Name = "lblLevel";
			this.lblLevel.Size = new System.Drawing.Size(280, 32);
			this.lblLevel.TabIndex = 3;
			this.lblLevel.Text = "level";
			// 
			// MainForm
			// 
			this.AutoScaleBaseSize = new System.Drawing.Size(6, 15);
			this.ClientSize = new System.Drawing.Size(584, 524);
			this.Controls.Add(this.lblLevel);
			this.Controls.Add(this.lblStatus);
			this.Controls.Add(this.lb);
			this.Controls.Add(this.prelimBoard);
			this.Menu = this.mainMenu1;
			this.Name = "MainForm";
			this.SizeGripStyle = System.Windows.Forms.SizeGripStyle.Show;
			this.Text = "LP Sokoban";
			this.ResumeLayout(false);

		}
		#endregion

		/// <summary>
		/// The main entry point for the application.
		/// </summary>
		[STAThread]
		static void Main() {
			try {
				Application.Run(new MainForm());
			} catch(Exception exc) {
				Console.WriteLine(exc.ToString());
			}
		}
#endregion

#region utilities
		//=================================================================
		// utilities
		//=================================================================
		// logs a one-line message
		private void log(string s) {
			// send to console (could be "Output" window in Visual Studio)
			Console.WriteLine(s);
			// also send to the built-in debug listbox (see View menu to enable/disable)
			lb.Items.Add(s); 
			// maximum scroll to see most recent lines, unless mouse is pressed
			if (Control.MouseButtons==MouseButtons.None) lb.TopIndex=lb.Items.Count-1;
		}

		// logs an exception
		private void log(Exception exc) {
			// let every CR or LF character be the start of a new line
			string[] sa=exc.ToString().Split('\n', '\r');
			// now only process non-empty lines
			foreach (string s in sa) if (s.Length!=0) log(s);
		}

		// gets the string value of a registry key (create it if it does not exist)
		private static string GetRegistryKey(string name, string defaultValue) {
			RegistryKey key=Registry.CurrentUser;
			key=key.CreateSubKey(REGISTRY_ROOT);
			return (string)key.GetValue(name, defaultValue);
		}

		// gets the int value of a registry key (create it if it does not exist)
		private static int GetRegistryKey(string name, int defaultValue) {
			RegistryKey key=Registry.CurrentUser;
			key=key.CreateSubKey(REGISTRY_ROOT);
			try {
				string s=(string)key.GetValue(name);	// null if non-existing
				// would use TryParse on .NET 2.0 !
				return int.Parse(s);
			} catch {
				return defaultValue;
			}
		}

		// sets the value of a registry key(create it if it does not exist)
		private static void SetRegistryKey(string name, string newValue) {
			RegistryKey key=Registry.CurrentUser;
			key=key.CreateSubKey(REGISTRY_ROOT);
			key.SetValue(name, newValue);
		}

		// sets the value of a registry key(create it if it does not exist)
		private static void SetRegistryKey(string name, int newValue) {
			SetRegistryKey(name, newValue.ToString());	// always store a string !
		}

		// sets the value of a registry key(create it if it does not exist)
		private static void SetRegistryKey(string name, bool newValue) {
			SetRegistryKey(name, newValue.ToString());	// always store a string !
		}

		// sets a new program state
		private void setState(string state) {
			lblStatus.Text=state;
		}

		[DllImport("user32.dll")]
		public static extern bool MessageBeep(int type);
		
#endregion

#region formEvents
		//=================================================================
		// form-related event handlers
		//=================================================================

		private void MainForm_Load(object sender, EventArgs e) {
			// apply persistent values from registry (or default ones)

			// form size
			Width=GetRegistryKey(REGISTRY_FORMWIDTH, Width);
			Height=GetRegistryKey(REGISTRY_FORMHEIGHT, Height);

			// toggle menus
			if (GetRegistryKey(REGISTRY_DEBUGFLAG, "False")!="False") menuDebug.PerformClick();
			
			// autoreload former puzzle
			string fileSpec=GetRegistryKey(REGISTRY_CURRENTFILE, null);
			int level=GetRegistryKey(REGISTRY_CURRENTLEVEL, 0);
			LoadPuzzle(fileSpec, level);
		}


		private void LoadPuzzle(string fileSpec, int level) {
			if (fileSpec!=null && level!=0) {
				try {
					sokoban.LoadLevels(fileSpec);
					sokoban.LoadLevel(level);
					this.fileSpec=fileSpec;
					this.level=level;
					SetRegistryKey(REGISTRY_CURRENTFILE, fileSpec);
					SetRegistryKey(REGISTRY_CURRENTLEVEL, level);
					Text=PROGRAM_NAME+" - "+Path.GetFileName(fileSpec)+" - level "+level;
					lblLevel.Text="Level "+level;
					int maxMoves=sokoban.MaxMoves;
					if (maxMoves!=int.MaxValue) lblLevel.Text="Level "+level+" (max "+maxMoves+" moves)";
					menuRestart_Click(null, null);
					return;
				} catch(Exception exc) {
					log(exc);
				}
			}
			// we get here if no file selected or I/O problems
			Text=PROGRAM_NAME;
			setState("Please open a Sokoban file");
			lblLevel.Text="";
		}

		private void MainForm_Resize(object sender, EventArgs e) {
			bool showListBox=menuDebug.Checked;
			int width=ClientRectangle.Width;
			int height=ClientRectangle.Height;
			int margin=prelimBoard.Bounds.X;
			int wmax;
			if (showListBox) {
				wmax=(width-3*margin)/2;	// margin panel margin listbox margin
			} else {
				wmax=(width-2*margin)/1;	// margin panel margin
			}
			int hmax=height-prelimBoard.Bounds.Y-margin;
			board.Bounds=new Rectangle(margin, board.Bounds.Y, wmax, hmax);
			lb.Bounds=new Rectangle(wmax+2*margin, lb.Bounds.Y, wmax, hmax);
			lb.Visible=showListBox;
		}

		private void MainForm_KeyDown(object sender, KeyEventArgs e) {
			Move move=null;
			switch (e.KeyCode) {
				case Keys.Up:
					move=sokoban.TryMove(0, -1, true);
					break;
				case Keys.Down:
					move=sokoban.TryMove(0, 1, true);
					break;
				case Keys.Left:
					move=sokoban.TryMove(-1, 0, true);
					break;
				case Keys.Right:
					move=sokoban.TryMove(1, 0, true);
					break;
				case Keys.Escape:
					pasteAbort=true;
					break;
			}
			postProcessMove(move);
			e.Handled=true;	// dont forward key event to a Control (e.g. dont scroll listbox)
		}

		private void postProcessMove(Move move) {
			if (move!=null) {
				board.Repaint();
				int moveCount=sokoban.MoveCount;
				int maxMoves=sokoban.MaxMoves;
				if (sokoban.Solved) {
					if (moveCount<=maxMoves) setState("Solved in "+moveCount+" moves");
					else setState("You did it, but too many moves ("+moveCount+")");
				} else {
					setState(moveCount==1?"1 move":moveCount.ToString()+" moves");
				}
				if (moveCount>maxMoves) MessageBeep(0x10);
				menuUndo.Enabled=sokoban.CanUndo;
				menuRedo.Enabled=sokoban.CanRedo;
			}
		}

		private void MainForm_Closed(object sender, EventArgs e) {
			SetRegistryKey(REGISTRY_FORMWIDTH, Width);
			SetRegistryKey(REGISTRY_FORMHEIGHT, Height);
		}
#endregion
		
#region menuEvents
		//=================================================================
		// menu-related event handlers
		//=================================================================

		private void menuOpen_Click(object sender, System.EventArgs e) {
			OpenFileDialog dialog=new OpenFileDialog();
			dialog.Filter="XML files|*.xml|All files|*.*";
			dialog.FileName=fileSpec;
			if (DialogResult.OK==dialog.ShowDialog()) {
				LoadPuzzle(dialog.FileName, 1);
			}
		}

		private void menuRestart_Click(object sender, System.EventArgs e) {
			sokoban.Restart();
			menuUndo.Enabled=false;
			menuRedo.Enabled=false;
			setState("Awaiting your first move");
		}

		private void menuNextLevel_Click(object sender, System.EventArgs e) {
			LoadPuzzle(fileSpec, ++level);
		}

		private void menuPreviousLevel_Click(object sender, System.EventArgs e) {
			LoadPuzzle(fileSpec, --level);		
		}

		private void menuPrint_Click(object sender, System.EventArgs e) {
			board.Print();
		}

		private void menuQuit_Click(object sender, System.EventArgs e) {
			Close();
		}

		private void menuUndo_Click(object sender, System.EventArgs e) {
			Move move=sokoban.UndoMove();
			postProcessMove(move);
		}


		private void menuRedo_Click(object sender, System.EventArgs e) {
			Move move=sokoban.RedoMove();
			postProcessMove(move);
		}

		private void menuCopy_Click(object sender, System.EventArgs e) {
			string s=sokoban.Solution;
			if (s!=null) {
				log("Copying "+s);
				Clipboard.SetDataObject(s, true);
			}
		}

		private void menuPaste_Click(object sender, System.EventArgs e) {
			IDataObject data=Clipboard.GetDataObject();
			if (data.GetDataPresent(DataFormats.Text)) {
				pasteString=data.GetData(DataFormats.Text).ToString();
				menuPaste.Enabled=false; // prevent multiple execution
				log("Pasting "+pasteString);
				// we want to introduce delays without inhibiting the normal
				// responsiveness of the UI; we could do that with a
				// System.Windows.Forms.Timer, but this time we use
				// a background thread (so we can show how to use
				// InvokeRequired and Invoke later on !)
				Thread thread=new Thread(new ThreadStart(pasteSequencer));
				thread.IsBackground=true;
				thread.Start();
			}
		}

		private void pasteSequencer() {
			string s=pasteString.ToUpper();
			pasteAbort=false;
			foreach (char c in s) {
				if (pasteAbort) break;
				pasteOneMove(c);
				// slow down for animation (more on SHFT, but not on ALT)
				if (Control.ModifierKeys!=Keys.Alt) {
					Thread.Sleep(100);
					if (Control.ModifierKeys==Keys.Shift) Thread.Sleep(1000);
				}
			}
			pasteOneMove(char.MaxValue);	// last paste action (reenable menu)
		}

		// we need a delegate to go back from the background thread to the UI thread
		private delegate void MovePaster(char c);

		private void pasteOneMove(char c) {
			if (InvokeRequired) {
				Invoke(new MovePaster(pasteOneMove), new object[]{c});
			} else {
				if (c==char.MaxValue) {
					menuPaste.Enabled=true;
				} else {
					Move move=null;
					if (c=='U') move=sokoban.TryMove(0,-1,true);
					else if (c=='D') move=sokoban.TryMove(0,1,true);
					else if (c=='L') move=sokoban.TryMove(-1,0,true);
					else if (c=='R') move=sokoban.TryMove(1,0,true);
					else if (c==' ') {}
					else if (c=='\t') {}
					else if (c=='\n') {}
					else if (c=='\r') {}
					else log("*** bad character in paste: "+c);
					if (move!=null) postProcessMove(move);
				}
			}
		}

		private void menuDebug_Click(object sender, System.EventArgs e) {
			bool debugging=!menuDebug.Checked;
			menuDebug.Checked=debugging;
			SetRegistryKey(REGISTRY_DEBUGFLAG, debugging);
			// first make sure the aspect ratio is OK
			if (debugging && Width<2*Height) Size=new Size(2*Height, Height);
			OnResize(null);	// this will fire MainForm_Resize() anyway
		}

		private void menuRules_Click(object sender, System.EventArgs e) {
			RulesForm rules=new RulesForm();
			rules.ShowDialog(this);
		}

		private void menuAbout_Click(object sender, System.EventArgs e) {
			AboutForm about=new AboutForm();
			about.ShowDialog(this);
		}
#endregion
	}
}
