//*******************************************************************
// LP Sokoban
//
// my implementation of the popular Sokoban puzzle
// Copyright: Luc Pattyn, January 2007
//
// This code is freely available for any non-commercial use.
//
//*******************************************************************

using System;
using System.Collections;			// ArrayList
using System.Text;					// StringBuilder
using System.Xml;

namespace Sokoban {

	/// <summary>
	/// Item represents one item on the board. It uses logical coordinates,
	/// not physical ones.
	/// </summary>
	public class Item {
		public ItemType type;
		private int xOriginal;
		private int yOriginal;
		public int x;
		public int y;

		public Item(int x, int y, ItemType type) {
			this.xOriginal=x;
			this.yOriginal=y;
			this.type=type;
			Restart();
		}
		public void Restart() {
			x=xOriginal;
			y=yOriginal;
		}
		public void Move(int dx, int dy) {
			this.x+=dx;
			this.y+=dy;
		}
	}

	/// <summary>
	/// The different item types we use on a board.
	/// </summary>
	public enum ItemType {
		Wall,
		Box,
		Goal,
		Sokoban
	}

	/// <summary>
	/// Move represents one move; it helps implementing the undo/redo mechanism.
	/// </summary>
	public class Move {
		public int dx;
		public int dy;
		public Item boxPushed;
		public Move(int dx, int dy, Item boxPushed) {
			this.dx=dx;
			this.dy=dy;
			this.boxPushed=boxPushed;
		}
		public override string ToString() {
			return "("+dx+","+dy+")"+(boxPushed==null?"":" push");
		}
	}

	/// <summary>
	/// Sokoban represents the puzzle input/output and logic.
	/// </summary>
	public class Sokoban {
		private Logger log;
		private XmlNodeList Levels;
		private int levelNumber;
		private string levelName;
		private int levelWidth;		public int LevelWidth {get {return levelWidth;}}
		private int levelHeight;	public int LevelHeight {get {return levelHeight;}}
		private int maxMoves;		public int MaxMoves {get {return maxMoves;}}
		private ArrayList walls=new ArrayList();	
									public ArrayList Walls {get {return walls;}}
		private ArrayList boxes=new ArrayList(); 
									public ArrayList Boxes {get {return boxes;}}
		private ArrayList goals=new ArrayList();
									public ArrayList Goals {get {return goals;}}
		private Item me=new Item(0,0, ItemType.Sokoban);
									public Item Me {get {return me;}}
		private int emptyGoals;		public bool Solved {get {return emptyGoals==0;}}
		private int lastDx=1;		public int LastDx {get {return lastDx;}}
		private int lastDy=0;		public int LastDy {get {return lastDy;}}
		private History history;
		private StringBuilder solution; public string Solution{get{return solution.ToString();}}

		public Sokoban(Logger log) {
			this.log=log;
			history=new History(log);
			solution=new StringBuilder();
		}

		public void LoadLevels(string fileSpec) {
			XmlDocument doc=new XmlDocument();
			doc.Load(fileSpec);
			Levels=doc.SelectNodes("//Level");
			log("Found "+Levels.Count+" levels");
		}

		public void LoadLevel(int level) {
			XmlNode Level=Levels[level-1];
			XmlAttributeCollection xac=Level.Attributes;
			levelNumber=level;
			levelName=xac["Id"].Value;
			levelWidth=int.Parse(xac["Width"].Value);
			levelHeight=int.Parse(xac["Height"].Value);
			maxMoves=int.MaxValue; // is optional field, so Value may return null
			try {maxMoves=int.Parse(xac["MaxMoves"].Value);}catch{}
			log("levelNumber="+levelNumber);
			log("levelName="+levelName);
			log("levelDimensions="+levelWidth+"*"+levelHeight);
			log("maxMoves="+maxMoves);
			XmlNodeList layout=Level.SelectNodes("L");
			goals.Clear();
			boxes.Clear();
			walls.Clear();
			me=null;
			for (int y=0; y<levelHeight; y++) {
				string line=layout[y].InnerText;
				for (int x=0; x<levelWidth; x++) {
					char c=' ';
					if (x<line.Length) c=line[x];
					switch (c) {
						case ' ':	// empty cell
							break;
						case '#':	// wall
							walls.Add(new Item(x, y, ItemType.Wall));
							break;
						case '$':	// box
							boxes.Add(new Item(x, y, ItemType.Box));
							break;
						case '.':	// goal
							goals.Add(new Item(x, y, ItemType.Goal));
							break;
						case '@':	// sokoban
							me=new Item(x, y, ItemType.Sokoban);
							break;
						case '*':	// box on goal
							boxes.Add(new Item(x, y, ItemType.Box));
							goals.Add(new Item(x, y, ItemType.Goal));
							break;
						case '+':	// sokoban on goal
							me=new Item(x, y, ItemType.Sokoban);
							goals.Add(new Item(x, y, ItemType.Goal));
							break;
						case '=':	// space (outside puzzle)
							break;
					}
				}
			}
			Restart();
		}

		public void Restart() {
			// move all boxes to original position, and recount the number of empty goals
			emptyGoals=0;
			foreach (Item item in boxes) {
				item.Restart();
				if (!IsOnGoal(item.x, item.y)) emptyGoals++;
			}
			me.Restart();	// move the sokoban to its original position
			history.Clear();
			solution.Length=0;
			log("boxes="+boxes.Count+", goals="+goals.Count+", emptyGoals="+emptyGoals);
		}

		// is there a goal at (x,y) ?
		public bool IsOnGoal(int x, int y) {
			foreach (Item item in goals) if (item.x==x && item.y==y) return true;
			return false;
		}

		// is there a wall or box at (x,y) ?
		private Item GetItem(int x, int y) {
			foreach (Item item in walls) {
				if (item.x==x && item.y==y) return item;
			}
			foreach (Item item in boxes) {
				if (item.x==x && item.y==y) return item;
			}
			return null;
		}

		// try to move in (dx,dy) direction
		// when redoing a move, set undoable to false
		public Move TryMove(int dx, int dy, bool undoable) {
			Move move=null;
			if (emptyGoals==0) {
				log("Can not move; puzzle was solved");
			} else {
				lastDx=dx;
				lastDy=dy;
				int x1=me.x;
				int y1=me.y;
				int x2=x1+dx;
				int y2=y1+dy;
				Item item2=GetItem(x2, y2);
				if (item2==null) {
					// move into an empty space (or goal)
					moveSokoban(dx, dy);
					move=new Move(dx, dy, null);
				} else if (item2.type==ItemType.Box) {
					Item item3=GetItem(x2+dx, y2+dy);
					if (item3==null) {
						// push a box into an empty space (or goal)
						moveSokoban(dx, dy);
						moveBox(item2, dx, dy);
						move=new Move(dx, dy, item2);
					}
				}
			}
			if (move!=null) {
				if (undoable) history.Do(move);
				solution.Append("LU.DR"[dx+dx+dy+2]);
				log(solution.ToString());
			}
			return move;
		}

		private void moveSokoban(int dx, int dy) {
			// moveCount not yet incremented !
			//log("Move "+(MoveCount+1)+": "+dx+" "+dy);
			me.Move(dx, dy);
		}

		private void moveBox(Item item, int dx, int dy) {
			if (IsOnGoal(item.x, item.y)) emptyGoals++;	// leaving a goal ?
			item.Move(dx, dy);
			if (IsOnGoal(item.x, item.y)) emptyGoals--;	// reaching a goal ?
			if (emptyGoals==0) {
				// moveCount not yet incremented !
				if (MoveCount<maxMoves) log("Solved!");
				else log("You did it, but in too many moves");
			}
		}

		public bool CanUndo {get {return history.CanUndo;}}
		public bool CanRedo {get {return history.CanRedo;}}

		public Move UndoMove() {
			Move move=(Move)history.Undo();
			Item box=move.boxPushed;
			lastDx=move.dx;
			lastDy=move.dy;
			if (box!=null) moveBox(box, -lastDx, -lastDy);
			moveSokoban(-lastDx, -lastDy);
			solution.Length=MoveCount;
			return move;
		}

		public Move RedoMove() {
			Move move=(Move)history.Redo();
			TryMove(move.dx, move.dy, false);
			return move;
		}

		public int MoveCount {get {return history.Index;}}
	}
}