﻿// LPTextFileDiffTester
// version V1.1 (August 2009)
// by Luc Pattyn, http://www.perceler.com/
//
// a small app to test the LPTextFileDiff class

using System;
using System.Diagnostics;			// Process, Stopwatch
using System.Drawing;				// Graphics
using System.IO;					// Directory, File, Path
using System.Threading;				// Thread
using System.Windows.Forms;			// Application

namespace LPMeanAndLean {
	public class LPTextFileDiffTester {
		static StreamWriter streamWriter=null;

		static void Main(string[] args) {
			string outputFilename=null;
			string[] fileNames=new string[2];
			int windowSize=30;
			int threshold=10;
			int fileCount=0;
			bool argError=false;
			bool logMatchingLines=false;
			bool autoQuit=true;
			bool gcCollect=true;
			bool createBigFiles=false;
			foreach (string arg in args) {
				if (arg.StartsWith("-") || arg.StartsWith("/")) {
					string option=arg.Substring(1).ToUpper();
					if (option=="Q") autoQuit=false;
					else if (option=="B") createBigFiles=true;
					else if (option=="S") gcCollect=false;
					else if (option=="M") logMatchingLines=true;
					else if (option=="N" || option=="M") outputFilename=Environment.GetFolderPath(
						Environment.SpecialFolder.ApplicationData)+@"\LPTextFileDiff\diff"+
						DateTime.Now.ToString("_yyyyMMdd_HHmmss")+".txt";
					else if (option.StartsWith("W")) {
						int w=0;
						if (int.TryParse(option.Substring(1), out w) && w>=5) windowSize=w;
						else argError=true;
					}  else if (option.StartsWith("T")) {
						int t=0;
						if (int.TryParse(option.Substring(1), out t) && t>=0) threshold=t;
						else argError=true;
					} else argError=true;
				} else {
					if (fileCount<2) fileNames[fileCount++]=arg;
					else argError=true;
				}
			}
			try {
				if (argError || fileCount!=2) {
					output("LPTextFileDiff version V1.0, by Luc Pattyn, August 2009");
					output("Compares two text files");
					output("");
					output("usage:  LPTextFileDiff [/option ...] file1 file2");
					output("");
					output("options:");
					output("\t/B   = create big files (expand by 10, 100, 1000)");
					output("\t/N   = generate an output file and open it with Notepad");
					output("\t/M   = also list matching lines to the output file");
					output("\t/Q   = wait for the user to hit a key before quitting");
					output("\t/S   = optimize for speed (no GC.Collect)");
					output("\t/Tdd = change threshold to dd matching lines (minimum 1, default 10)");
					output("\t/Wdd = change initial sliding window size to dd lines (min 5, def 30)");
					output("");
				} else {
					if (outputFilename!=null) {
						Directory.CreateDirectory(Path.GetDirectoryName(outputFilename));
						streamWriter=File.CreateText(outputFilename);
					}
					if (createBigFiles) {
						foreach (string fin in fileNames) {
							string insert="";
							for (int factor=10; factor<=1000; factor*=10) {
								insert="0"+insert;
								string fout=fin.Replace(".", insert+".");
								string[] data=File.ReadAllLines(fin);
								StreamWriter sw1000=File.CreateText(fout);
								for (int i=0; i<factor; i++) {
									foreach (string s in data) sw1000.WriteLine(s);
								}
								sw1000.Close();
								output("Created file "+fout);
							}
						}
						Application.Exit();
					}
					Thread.Sleep(100);
					Thread.CurrentThread.Priority=ThreadPriority.AboveNormal;
					Stopwatch sw1=new Stopwatch();
					sw1.Start();
					LPTextFileDiff comparer=new LPTextFileDiff(null, null, 0, 0, false, null, false);
					comparer.Compare(); // precompile
					sw1.Stop();
					//msec=0.1*(ticks/1000);
					output("JITted in "+sw1.Elapsed.TotalMilliseconds.ToString("N1")+" msec");
					output("File1="+fileNames[0]);
					output("File2="+fileNames[1]);
					for (int i=0; i<3; i++) GC.Collect();	// get a stable GC situation before launching the test
					Process process = Process.GetCurrentProcess();
					process.Refresh();
					long startPeakWorkingSet64=process.PeakWorkingSet64;
					Stopwatch sw2=new Stopwatch();
					sw2.Start();
					comparer=new LPTextFileDiff(fileNames[0], fileNames[1], windowSize, threshold, gcCollect,
						streamWriter, logMatchingLines);
					string report=comparer.Compare();
					sw2.Stop();
					process.Refresh();
					long deltaPeakWorkingSet64=process.PeakWorkingSet64-startPeakWorkingSet64;
					output(report);
					output("");
					output("LPTextFileDiff done (using "+sw2.Elapsed.TotalMilliseconds.ToString("N1")+
					" msec and a working set increase of "+((deltaPeakWorkingSet64+512)/1024)+" KB)");
				}
			} catch (Exception exc) {
				foreach (string s in exc.ToString().Split('\r', '\n')) if (s.Length!=0) output(s);
			} finally {
				if (streamWriter!=null) {
					streamWriter.Close();
					streamWriter=null;
					output("Output file generated:");
					output(outputFilename);
					Process.Start("notepad.exe", outputFilename);
				}
			}
			if (!autoQuit) {
				Console.WriteLine("");
				Console.WriteLine("Hit ENTER to terminate...");
				Console.ReadKey();
			}
		}

		// output to both console and file (iff open)
		static void output(string s) {
			if (streamWriter!=null) streamWriter.WriteLine(s);
			Console.WriteLine(s);
		}
	}
}
